package edu.ucla.ccb.graphshifts.image;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Random;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;

/**
 * The Colormap class is just a simple class to store the packed int RGB colormaps
 *  that are used in my imaging display code.  I made a class so that the creation
 *  routines can all be stored in the same place, and there is some self-contained way
 *  to allow editing of the colormap and display in its own window (which it will manage).
 * @author jcorso
 *
 */
public class Colormap
{
	int map[];
	
	/** a colormap cannot just be created by the used.  It must be created by
	 *  one of the static members.
	 *
	 */
	protected Colormap()
	{
	}
	protected Colormap(int n)
	{
		map = new int[n];
	}
	
	
	public int getLength()
	{
		return map.length;
	}
	public int getColorAsIntARGB(int i)
	{
		return map[i];
	}
	public Color getColor(int i)
	{
		return new Color(map[i]);
	}
	
	
	public JMenuItem createInspectorMenuItem(String name)
	{
		JMenuItem item = new JMenuItem(name);
		item.addActionListener(new ColormapInspectorMenuItemActionListener(this));
		return item;
	}
	
	
	public JDialog showInspector()
	{
		return new DefaultColormapInspector(this);
	}
	
	
	////    STATIC CREATION ROUTINES
	
	/** File is assumed to be a whitespace separate list of # rgb tuples */
	static public Colormap createFromFile(String fn)
	{
		Colormap C = new Colormap(256);
		
		try
		{
			BufferedReader rdr = new BufferedReader( new FileReader(fn) );
			
			int i=0;
			int r,g,b;
			int alpha=255;
			while (i<256 && rdr.ready())
			{
				String L = rdr.readLine();
				String A[] = L.split("\\s");
				r = Integer.parseInt(A[1]);
				g = Integer.parseInt(A[2]);
				b = Integer.parseInt(A[3]);
				C.map[i++] = ((alpha<<24) | (r<<16) | (g<<8) | b);
			}
			
			rdr.close();
		} catch (IOException e)
		{
			System.err.println("Could not read the colormap in "+fn+", generating a random one instead\n");
			return createRandom(256);
		}
		
		return C;
	}
	
	static public Colormap createFromMap(int[] map)
	{
		Colormap C = new Colormap();
		C.map = map;
		return C;
	}
	static public Colormap createLinear(int n, int startColor, int endColor, int alpha)
	{
		return createLinear(n,new Color(startColor),new Color(endColor),alpha);
	}
			
	static public Colormap createLinear(int n, Color startColor, Color endColor, int alpha)
	{
		Colormap C = new Colormap(n);

		float A[] = new float[]{startColor.getRed(),startColor.getGreen(),startColor.getBlue()};
		float B[] = new float[]{endColor.getRed(),endColor.getGreen(),endColor.getBlue()};
		float D[] = new float[3];
		for (int i=0;i<3;i++)
			D[i] = B[i]-A[i];
		
		float del = 1.0f/(float)n;
	
		int r,g,b;
		for(int i=0;i<n;i++)
		{
			r = (int) (A[0]+D[0]*(float)i*del);
			g = (int) (A[1]+D[1]*(float)i*del);
			b = (int) (A[2]+D[2]*(float)i*del);
			C.map[i] = ((alpha<<24) | (r<<16) | (g<<8) | b);
		}
		
		return C;
	}
	static public Colormap createRandom(int n)
	{
		return createRandom(n,255);
	}
	static public Colormap createRandom(int n, int alpha)
	{
		Colormap C = new Colormap(n);
		
		Random rand = new Random();
		
		int r,g,b;
		for(int i=0;i<n;i++)
		{
			r = rand.nextInt(255);
			g = rand.nextInt(255);
			b = rand.nextInt(255);
			//pack these bytes into colormap_[i]
			C.map[i] = ((alpha<<24) | (r<<16) | (g<<8) | b);
		}
		return C;
	}
	
	
	
	/////  NESTED PRIVATE CLASSES
	private class ColormapInspectorMenuItemActionListener implements ActionListener 
	{
		Colormap map;
		
		public ColormapInspectorMenuItemActionListener(Colormap map)
		{
			this.map = map;
		}
		
		public void actionPerformed(ActionEvent e)
		{
			System.out.println("inspecting colormap");
			map.showInspector();
		}
		
	}
	
	
	private class DefaultColormapInspector extends JDialog
	{
		Colormap map;
		public DefaultColormapInspector(Colormap map)
		{
			this.map = map;
			this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
			this.setPreferredSize(new Dimension(320,240));
		
			buildDialog();

			this.setTitle("Colormap Inspector");
			this.pack();
			this.setVisible(true);
		}
		
		private void buildDialog()
		{
			setLayout(new BorderLayout());
			
			// first thing we do is build a horizontal panel
			JPanel colorPanel = new JPanel();
			colorPanel.setPreferredSize(new Dimension(300,60));
			colorPanel.setLayout(new BoxLayout(colorPanel,BoxLayout.X_AXIS));
			colorPanel.setBorder(BorderFactory.createEmptyBorder(5,10,5,10));
			
			// now, add each of the colors to the panel as a subPanel
			Dimension wellD = new Dimension( 300 / map.getLength(), 60);
			JPanel well;
			JLabel wellLabel;
			Font f = new Font("Sans Serif",Font.PLAIN,8);
			for (int i=0;i<map.getLength();i++)
			{
				well = new JPanel();
				well.setPreferredSize(wellD);
				//well.setMaximumSize(wellD);
				//well.setMinimumSize(wellD);
				well.setBackground(map.getColor(i));
				wellLabel = new JLabel(Integer.toString(i));
				wellLabel.setFont(f);
				well.add(wellLabel);
				colorPanel.add(well);
			}
			add(colorPanel,BorderLayout.NORTH);
		}
		
	}
}
