package edu.ucla.ccb.graphshifts;

import gnu.trove.TIntArrayList;
import gnu.trove.TIntHashSet;
import gnu.trove.TIntIntHashMap;
import gnu.trove.TIntIntIterator;
import gnu.trove.TLongByteHashMap;
import gnu.trove.TLongHashSet;
import gnu.trove.TLongIterator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Random;

import edu.ucla.ccb.graphshifts.graphs.HardGraphHierarchy;
import edu.ucla.ccb.graphshifts.graphs.ThinGraph;
import edu.ucla.ccb.graphshifts.graphs.ThinGraph.ThinGraphNode;
import edu.ucla.ccb.graphshifts.image.AnalyzeHeader;
import edu.ucla.ccb.graphshifts.image.Colormap;
import edu.ucla.ccb.graphshifts.image.Image3PixelAccess;
import edu.ucla.ccb.graphshifts.image.Image3Protocol;
import edu.ucla.ccb.graphshifts.image.ImageUtilities;
import edu.ucla.ccb.graphshifts.image.JIUIntegerImage_Image3PixelAccess_Adaptor;
import edu.ucla.ccb.graphshifts.image.ScalarImageB3;
import edu.ucla.ccb.graphshifts.data.SingleFloatArrayDataItem;
import edu.ucla.ccb.graphshifts.observers.GraphShiftsObserver;


import net.sourceforge.jiu.codecs.ImageLoader;
import net.sourceforge.jiu.color.reduction.RGBToGrayConversion;
import net.sourceforge.jiu.data.IntegerImage;
import net.sourceforge.jiu.data.MemoryRGB24Image;
import net.sourceforge.jiu.data.PixelImage;
import net.sourceforge.jiu.data.RGB24Image;

/**
 * The first implementation of the GraphShifts algorithm for general use.  It's the graph-shifts
 *  algorithm with split/merge and spawn moves.  The energy is the one used in the IPMI submission 
 *  and the code requires a set of likelihood maps for each of the models.
 *  
 *  Updated 20070712 to handle multiline files for the maps input.
 * @author jcorso
 *
 */

public class GraphShifts 
{
	
	static public interface FixedModelBinaryEnergyComputer
	{
		public double computeEnergy(int length, int modelA, int modelB);
	}
	
	static public class BECSmoothness implements FixedModelBinaryEnergyComputer
	{
		public double computeEnergy(int length, int modelA, int modelB)
		{
			return length * (1 - ((modelA==modelB)?1:0) );
		}
	}
	
	static public class BECContext implements FixedModelBinaryEnergyComputer
	{
		// Energy is doubly indexed by modelA and modelB
		double energy[][];
		
		public BECContext(double[][] energy)
		{
			this.energy = energy;
		}
		
		/** first line of the file is the dimensionality of the matrix */
		public BECContext(BufferedReader br)
		{
			String line;
			
			try 
			{
				line = br.readLine();
				int dim = Integer.parseInt(line);
				energy = new double[dim][dim];

				for (int i=0;i<dim;i++)
				{
					line = br.readLine();
					String[] vals = line.split("\\s");
					for (int j=0;j<dim;j++)
					{
						energy[i][j] = Double.parseDouble(vals[j]);
					}
				}
			}
			catch (IOException e)
			{
				System.err.println("IOException while reading the BEC context energy matrix");
				e.printStackTrace();
			}
		}
		
		public double computeEnergy(int length, int modelA, int modelB)
		{
			return length * energy[modelA][modelB];
		}
	}
	
	
	/**
	 * This enum specifies the different ways the image data argument (arg0) can
	 *  be specified by the user.
	 * @author jcorso
	 *
	 */
	static private enum ARG0TYPE
	{
		JUSTIMAGE,
		MULTILINE,
		COMMASEP
	}
	
	static public class EdgeSampleInitializer implements GraphShiftsInitializer
		{
			static final double kAlpha = -0.09;
			/** kTau stores the mean of the distances at the current level in the graph hierarchy.
			 * It is used during the edge "sampling" as the threshold.
			 */
			double cTau;
			
			// cache some of the variables that we work with.
			ImageData image;
			int w,h,d,wh;
			
			HardGraphHierarchy H;
			
//			String priorFN = "/data/Tu-lonidata/labels/prior";
//			Image3PixelAccess priorImage;
			
			LikelihoodMapSet lms;
			
			int levelMax;  // is used during initial coarsening to set the max number of levels
			               // in the hierarchy.  If it is 0, then there will only be a voxel layer
			               // and the top layer.
			double reductionFactor = 0.15;
			double lambda = 0.1;
			
			int minSpawnLevel=-1;
			
			
			
			public EdgeSampleInitializer (LikelihoodMapSet lms, int levelMax, double reductionFactor, double lambda,int minSpawnLevel)
			{
//				AnalyzeHeader hdr = new AnalyzeHeader(priorFN);
//				priorImage = hdr.loadFromDisk();
				
				this.lms = lms;
				
				this.levelMax = levelMax;
				
				this.reductionFactor = reductionFactor;
				
				this.lambda = lambda;
				
				this.minSpawnLevel = minSpawnLevel;
			}
			
			private ThinGraph coarsen(ThinGraph Gin, int level)
			{
				int num = Gin.getNumberOfNodes();
				if (num == 0)
					return null;
				
				ThinGraph Gout = new ThinGraph(Math.min(10000, num),1,1,numClasses);
				Random R = new Random(100L);
				
				// first we want to add the special spawn node at the beginning of the graph nodes
				//  to have index 0 (if it is at the minspawnlevel
				if (level >= minSpawnLevel)
				{
					Gout.addNewNode();
				}
				
				
				// this will help us manage what nodes have been looked at but not accepted as a part of
				//  the "current" node we're lifting
				TIntArrayList lookOnceFixer = new TIntArrayList(100);
				
				// this will threshold the largest possible region we are willing to create during coarsening
				//  at this level.  It is a function of the current number of nodes we have and the reduction
				//  factor we want.  
				
				int maxNodeSize = (int)(1.0 / reductionFactor); 
				
				// we use the parent field in the Gin graph to determine if a node has yet been touched
				//  which we assume is initialized to -1
			
				TIntArrayList S = new TIntArrayList(maxNodeSize); // this capacity is just a placeholder
				// loop through each node
				for (int i=0;i<num;i++)
				{
					// if the node has not yet been added to the next layer
					if (Gin.NDgetParent(i) == -1)
					{
						// now, we'll create a new node for the coarser layer in the graph
						ThinGraphNode C = Gout.addNewNode();
						int Cid = C.getId();
						
						lookOnceFixer.clear();
						int intensity = 0;
						int mass = 0;
						int size = 0;
						
						S.reset();
						S.add(i);
						while(!S.isEmpty())
						{
							int iv = S.remove(0);
							
							// we need to be adding this fine node to the coarse graph and linking
							//  up the parent child relationship
							Gin.NDsetParent(iv,Cid);
							Gout.NDaddChild(Cid, iv);
							
							intensity += Gin.NDgetIntensityUnsigned(iv);
							mass += Gin.NDgetMass(iv);
							size++;
							
							// if this new node became too large, break out of this loop 
							if (size >= maxNodeSize)
								break;
							
							// for all v's neighbors that are "on" add and not yet in CC,
							int mum = Gin.NDgetNumberOfNeighbors(iv);
							for (int j=0;j<mum;j++)
							{
								int iw = Gin.NDgetNeighbor(iv, j);
								if (Gin.NDgetParent(iw) == -1)
								{
									// check the edge weights...
									double data = Math.exp(kAlpha *
											      Math.abs((double)(Gin.NDgetIntensityUnsigned(iv)) - 
				    			                           (double)(Gin.NDgetIntensityUnsigned(iw))));
									// a linear combination between a uniform and the exponential edge presence
									data = lambda*0.5 + (1-lambda)*data;  
									if (R.nextDouble() < data)
									{
										// we need to set the parent to something other than -1 otherwise
										//  it can potentially get added to the queue multiple times....
										Gin.NDsetParent(iw, -2);
										S.add(iw);
									}
									else 
									{
										// we need to avoid testing a node more than 1 times or else
										//  we are more likely to take nodes than leave them...
										Gin.NDsetParent(iw, -3);
										lookOnceFixer.add(iw);
									}
								}
							}
						}
						
						// if we broke out of it because of size, then we need to fix the parents of all the
						//  nodes still in the queue back to -1 because they were never added
						int Sremaining = S.size();
						for (int iv=0;iv<Sremaining;iv++)
							Gin.NDsetParent(S.getQuick(iv), -1);
												
						C.setIntensity((byte)(intensity/size));
						C.setMass(mass);
					
						// now, we have to go back and turn any node that has a parent of -3 into a parent of -1
						int lOnceLen = lookOnceFixer.size();
						for (int j=0;j<lOnceLen;j++)
							Gin.NDsetParent(lookOnceFixer.get(j), -1);
					
					}
				}
				
				// now create neighboring edges in the new graph
				TIntHashSet hash = new TIntHashSet();
				int Goutn = Gout.getNumberOfNodes();
				for (int i=0;i<Goutn;i++)
				{
					int inn = Gout.NDgetNumberOfNeighbors(i);
					
					hash.clear(); // this hash stores if we already made this guy a neighbor
					// to avoid making double neighbors, we will begin by adding those ThinGraphNodes to which
					// this guy is already a neighbor (created when looking in the opposite direction)
				    hash.add(i);	
					for (int cn=0;cn<inn;cn++)
						hash.add(Gout.NDgetNeighbor(i,cn));
					
					int inc = Gout.NDgetNumberOfChildren(i);
					for (int ci=0;ci<inc;ci++)
					{
						int cid   = Gout.NDgetChild(i, ci);
						int cidnn = Gin.NDgetNumberOfNeighbors(cid);
						for (int cin=0;cin<cidnn;cin++)
						{
							int cidni = Gin.NDgetNeighbor(cid, cin);
							// now we have to check the parent index	
							int parent = Gin.NDgetParent(cidni);
						    if (!hash.contains(parent))
						    {
								Gout.addEdge(i,parent);
						    	hash.add(parent);
						    }
						}
					}
				}
				
				return Gout;
			}
					
			
			// accumulate node class membership based on the loaded appearance likelihood maps from Zhuowen 
			//  thus, this is a recursive function.  Note, the likelihood maps also take into account location
			private	void computeNodeClassHistogram_A(HardGraphHierarchy H,int level,int []clbuf, int node)
			{
				if (level == 0)
				{
					int x,y,z;
					z = ((int)Math.floor((double)node / (double)(wh)));
					y = ((int)Math.floor((double)(node-z*wh) / (double)w));
					x = ((node-z*wh) % w);
					// this can be improved to compute the max once, offline and load it
					int n = lms.classNum;
					int besti=0;
					int bestv=lms.maps[0].getPixelInt(x, y, z);
					for (int i=1;i<n;i++)
					{
						int v = lms.maps[i].getPixelInt(x, y, z);
						if (v > bestv)
						{
							bestv = v;
							besti = i;
						}
					}
					clbuf[besti]++;
					return;
				}
				
				ThinGraph G = (ThinGraph)H.getLayer(level);
				int cn = G.NDgetNumberOfChildren(node);
				for (int i=0;i<cn;i++)
					computeNodeClassHistogram_A(H,level-1,clbuf,G.NDgetChild(node, i));
			}

			// accumulate node class membership based on spatial location of its leaf voxels...
			//  thus, this is a recursive function
//			private	void computeNodeClassHistogram_S(HardGraphHierarchy H,int level,int []clbuf, int node)
//			{
//				if (level == 0)
//				{
//					int x,y,z;
//					z = ((int)Math.floor((double)node / (double)(wh)));
//					y = ((int)Math.floor((double)(node-z*wh) / (double)w));
//					x = ((node-z*wh) % w);
//					clbuf[priorImage.getPixelInt(x, y, z)]++;
//					return;
//				}
//				
//				ThinGraph G = (ThinGraph)H.getLayer(level);
//				int cn = G.NDgetNumberOfChildren(node);
//				for (int i=0;i<cn;i++)
//					computeNodeClassHistogram_S(H,level-1,clbuf,G.NDgetChild(node, i));
//			}
			
			private ThinGraph createTopLevelFull(HardGraphHierarchy H, ThinGraph g1)
			{
				/* create the top layer with one node per model */
				int gTopNum = numClasses;
				ThinGraph gTop = new ThinGraph(gTopNum,gTopNum,1,numClasses);
				H.addGraph(gTop);
				
				for (int i=0;i<gTopNum;i++)
				{
					ThinGraphNode n = gTop.addNewNode();
					n.setClassIndex(i);
				}
				
				/* go through g1 and link up each node to the toplayer node it most resembles */
				SingleFloatArrayDataItem fadi = new SingleFloatArrayDataItem();
				int idx;
				int count[] = new int[gTopNum];
				int clbuf[] = new int[numClasses]; // stores pixel counts for a given node to classes
				for (int i=0;i<g1.getNumberOfNodes();i++)
				{
					Arrays.fill(clbuf,0);
					computeNodeClassHistogram_A(H,H.getNumberOfLayers()-2,clbuf,i);
					
					idx=0;
				    for (int j=1;j<gTopNum;j++)
				    	idx = (clbuf[j] > clbuf[idx]) ? j : idx;
				    g1.NDsetClassIndex(i, idx);
				    count[idx]++;
				}
				
				/* check if each top-level node has at least one child */
				o.printf("checking if each top-level node has a child\n");
				for (int i=0;i<gTopNum;i++)
				{
					if (count[i] == 0)
					{
						o.printf("need to fix for index %d\n",i);
						/*
						// fix this by going through each node and taking the node that has
						//  the highest likelihood for this class -- even if it is not the highest
						//  likelihood that it already has
						// we're only premitted to take it, if it will not leave its parent with 0 
						Class c = scs.getClass(i);
						double bestLH=-1;
						ThinGraphNode bestN=null;
						o.printf("g1 has %d nodes\n",g1.getNumberOfNodes());
						for (int j=0;j<g1.getNumberOfNodes();j++)
						{
							ThinGraphNode n = (ThinGraphNode)g1.getNode(j);
							fadi.setData(n.getIntensityScaled());
							double LH = c.classify(fadi);
							if ((LH > bestLH) && (count[n.getClassIndex()]>1))
							{
								bestN = n;
								bestLH = LH;
							}
						}
						/* bestN should never be null here unless we have fewer nodes than classes which 
						 *  should not happen
						 *
						o.printf("best N is %s\n",bestN);
						count[bestN.getClassIndex()]--;
						count[i]++;
						bestN.setClassIndex(i);
						*/
					}
					o.printf("top layer %d has %d children\n",i,count[i]);
				}
				
				/* link up the top layer and g1 layer based on the class labels */
				/* can also percolate the class labels down the hierarchy here */
				o.printf("linking the top layers and pushing class labels down\n");
				for (int i=0;i<g1.getNumberOfNodes();i++)
				{
					// OLD
//					ThinGraphNode g1N = (ThinGraphNode)g1.getNode(i);
//					ThinGraphNode gTopN = (ThinGraphNode)gTop.getNode(g1N.getClassIndex());
//					
//					g1N.setParent(gTopN);
//					gTopN.addChild(g1N);
//					pushClassIndexDown(g1N,g1N.getClassIndex(),H,H.getNumberOfLayers()-2);
//					
					int topi = g1.NDgetClassIndex(i);
					g1.NDsetParent(i, topi);
					gTop.NDaddChild(topi, i);
					pushClassIndexDown(i,topi,H.getNumberOfLayers()-2);
					
				}
				
				return gTop;
			}
			
			
			
	
			public HardGraphHierarchy initializeGraph(ImageData image)
			{
				this.image = image;
				w  =  image.getWidth();
				h  =  image.getHeight();
				d  =  image.getDepth();
				wh =  w * h;
				
				o.printf("EdgeSampleInitializer Class:  creating lattice graph\n");
				//  XXXXXX  -- need to somehow get the variable dimension ThinGraph object....
				ThinGraph voxelGraph = ThinGraph.createFromVolume(image.getImage(0));
				o.printf("EdgeSampleInitializer Class:  done creating lattice graph\n");
				
				
				H = new HardGraphHierarchy();
				ThinGraph g1;
				
				g1 = voxelGraph;
				H.addGraph(g1);
				int level=1;
				System.out.printf("coarsening graph (%d nodes)\n",g1.getNumberOfNodes());
				
				if (levelMax != 0)  // if it is zero, we do no coarsening
				{
					while (g1.getNumberOfNodes() > 6*numClasses)
					{
						// "coarsen" g1 into g2.
						g1 = coarsen(g1,level++);
						H.addGraph(g1);
						System.out.printf("coarsening graph (%d nodes)\n",g1.getNumberOfNodes());

						if (H.getNumberOfLayers() > levelMax)
							break;
					}
				}
				
				
				/*** ----------------------  Hierarchy exists at this point  ----------------- ***/
				
				ThinGraph gTop = createTopLevelFull(H,g1);
				int gTopNum = gTop.getNumberOfNodes();
				
				/* we push up the node masses and other statistics here */
				for (int i=0;i<gTopNum;i++)
				{
					int mass=0;
//					ThinGraphNode gTopN = (ThinGraphNode)gTop.getNode(i);
					int nc = gTop.NDgetNumberOfChildren(i);
					for (int j=0;j<nc;j++)
					{
//						int ci = gTopN.getChild(j);
//						ThinGraphNode c = (ThinGraphNode)g1.getNode(ci);
//						mass += c.getMass();
						int ci = gTop.NDgetChild(i, j);
						mass += g1.NDgetMass(ci);
					}
//					gTopN.setMass(mass);
					gTop.NDsetMass(i, mass);
				}
				
				/* now, we can also compute the length of every boundary in the graph */
				/* we start with the second layer (since the first layer already knows its length is 1). */
				TIntIntHashMap iihm = new TIntIntHashMap(20);
				for (int i=1;i<H.getNumberOfLayers();i++)
				{
					ThinGraph gi  = (ThinGraph)H.getLayer(i);
					ThinGraph gi_ = (ThinGraph)H.getLayer(i-1);
					
					int gin = gi.getNumberOfNodes(); 
					
					for (int gid=0;gid<gin;gid++)
					{
						int cn = gi.NDgetNumberOfChildren(gid);
						iihm.clear();
						
						for (int cid=0;cid<cn;cid++)
						{
							int ci = gi.NDgetChild(gid, cid);
							int cin = gi_.NDgetNumberOfNeighbors(ci);
							
							for (int cini=0;cini<cin;cini++)
							{
								int nid = gi_.NDgetNeighbor(ci, cini);
								int pid = gi_.NDgetParent(nid);
							
								// if we have the same parent then buzz-off
								if (pid == gid)
									continue;

								int len = gi_.NDgetBoundaryLength(ci, cini);

								if (!iihm.adjustValue(pid,len))
									iihm.put(pid,len);
							}
						}   // end of looping through gid's children
						
						int nn = gi.NDgetNumberOfNeighbors(gid);
						
						for (int ni=0;ni<nn;ni++)
							gi.NDsetBoundaryLength(gid, ni, iihm.get( gi.NDgetNeighbor(gid, ni) ));
					}
					
				}
				iihm = null;
				
				
				/*** -------------------  Done Initializing Boundary Lengths ----   ------ ***/
				
                /*** -------------------  Initialize Spawn Node Connections -------------- ***/
				
				/************* CHANGE -- the spawn node connections are now initialized in the BMAPMS 
				 * initializeGraph function.  This makes it easy to check if the likelihood of a spawn is
				 * too little (and then not make the actual spawn connection).
				 *
				for (int i=minSpawnLevel;i<H.getNumberOfLayers()-1;i++)
				{
					ThinGraph gi  = (ThinGraph)H.getLayer(i);
					
					int gin = gi.getNumberOfNodes(); 
					for (int gid=1;gid<gin;gid++)
					{
						gi.addEdge(0, gid,0);
					}
				}
				*/
				
				/*** -------------------  Done Initializing Spawn Node Connections ------- ***/
			 
				
				/// below is for debug and not necessary in the real code
				/* here, we just output the mass of each of the toplevel nodes (mostly for
				 *  debugging purposes and this code can be removed in the future.
				 */
				/*
				int mass=0;
				for (int i=0;i<gTopNum;i++)
				{
					ThinGraphNode gTopN = (ThinGraphNode)gTop.getNode(i);
					o.printf("top layer node %d has mass of %d\n", i, gTopN.getMass());
					mass += gTopN.getMass();
				}
				o.printf("total mass at the top layer is %d\n",mass);
				o.printf("total number of voxels we started with %d\n",w*h*d);
				*/
				
				return H;
			}
			
			/* depth-first traversal implementation */
			private void pushClassIndexDown(int nodeIdx, int classIdx, int level)
			{
				ThinGraph G = (ThinGraph)H.getLayer(level);
				G.NDsetClassIndex(nodeIdx, classIdx);
				if (level == 0)
					return;
				
				int nc = G.NDgetNumberOfChildren(nodeIdx);
				for (int c=0;c<nc;c++)
				{
					int ci = G.NDgetChild(nodeIdx, c);
					pushClassIndexDown(ci,classIdx,level-1);
				}
			}
			
			/** old
			private void pushClassIndexDown(ThinGraphNode A,int classIdx,HardGraphHierarchy H, int cLevel)
			{
				A.setClassIndex(classIdx);
				if (cLevel == 0)
					return;
				
				ThinGraph finerG = (ThinGraph)H.getLayer(cLevel - 1);
				int clen = A.getNumberOfChildren();
				for (int i=0;i<clen;i++)
				{
					int ci = A.getChild(i);
					pushClassIndexDown((ThinGraphNode)finerG.getNode(ci),classIdx,H,cLevel - 1);
				}
			}
			*/
			
		}
	
	public interface GraphShiftsInitializer {
		public HardGraphHierarchy initializeGraph(ImageData image);
	}
	
	/**
	 * ImageData  (includes support for multichannel images...)
	 */
	static class ImageData 
	{
		static public ImageData createFromCSFile(String path, int dim)
		{
			String typespecifier[] = new String[1];
			
			ArrayList<String> names = new ArrayList<String>();
			
			BufferedReader in;
			String line;
			ImageData lidata=null;
			int w=-1,h=-1,d=-1;
			try 
			{
				in = new BufferedReader( new FileReader(path) );
				line = in.readLine(); 
				in.close();
				
				String args[] = line.split(",");
				
				for (int i=1;i<args.length;i++)
					names.add(args[i]);

				lidata = new ImageData(names.toArray(typespecifier));
			} 
			catch (FileNotFoundException e) 
			{
				e.printStackTrace();
			} 
			catch (IOException e) 
			{
				e.printStackTrace();
			}
			
			lidata.datadim = dim;
			
			return lidata;
		}
		
		
		static public ImageData createFromImage(String path, int dim)
		{
			String dummy[] = new String[1];
			dummy[0] = path;
			ImageData lidata = new ImageData(dummy);
			lidata.datadim = dim;
			
			return lidata;
		}
		
		static public ImageData createFromMLFile(String path, int dim)
		{
			String typespecifier[] = new String[1];
			
			ArrayList<String> names = new ArrayList<String>();
			
			BufferedReader in;
			String line;
			ImageData lidata=null;
			int w=-1,h=-1,d=-1;
			try 
			{
				in = new BufferedReader( new FileReader(path) );
				line = in.readLine(); // discard the first line
				line = in.readLine();
				while (line != null)
				{
					names.add(line);
					line = in.readLine();
				}
				in.close();

				lidata = new ImageData(names.toArray(typespecifier));
			} 
			catch (FileNotFoundException e) 
			{
				e.printStackTrace();
			} 
			catch (IOException e) 
			{
				e.printStackTrace();
			}
			
			lidata.datadim = dim;
			
			return lidata;
		}
		
		
		String[] imagePaths;
		Image3PixelAccess[] images;
		boolean loaded = false;
		float IRangeMin = 0.0f;
		float IRangeMax = 1.0f; 
		float IRangeScale = 1.0f;
		int dim;
		int w=-1,h,d;
		float weight;
		
		// specifies the dimensionality of the images (and thus the type of loading to do)
		int datadim=3;
		
		public ImageData(String p[])
		{
			imagePaths = p;
			images = new Image3PixelAccess[imagePaths.length];
			dim = imagePaths.length;
			weight = 1/dim;
		}
		public ImageData(String p[], float Imin, float Imax)
		{
			imagePaths = p;
			images = new Image3PixelAccess[imagePaths.length];
			dim = imagePaths.length;
			weight = 1/dim;
			this.IRangeMin = Imin;
			this.IRangeMax = Imax;
			this.IRangeScale = 1.0f / (IRangeMax - IRangeMin);
		}
		
		public int getDepth()
		{
			return d;
		}
		
		public int getDim()
		{
			return dim;
		}
		
		public int getHeight()
		{
			return h;
		}

		public Image3PixelAccess getImage(int i)
		{
			return images[i];
		}
		
		public int getWidth()
		{
			return w;
		}

		public boolean isDataLoaded()
		{
			return loaded;
		}
		
		public void loadData()
		{
			if (!loaded)
			{
				for (int i=0;i<images.length;i++)
				{
					System.out.printf("loading image %s\n", imagePaths[i]);
					if (datadim == 3)
					{
						AnalyzeHeader hdr = new AnalyzeHeader(imagePaths[i]);
						images[i] =  hdr.loadFromDisk();
					}
					else if (datadim == 2)
					{
						try
						{
							PixelImage pi = ImageLoader.load(imagePaths[i]);
							if (pi instanceof RGB24Image)
							{  // here, we'll need to convert the rgb image to grayscale
								RGBToGrayConversion rgbtogray = new RGBToGrayConversion();
								rgbtogray.setOutputImage(null);
								rgbtogray.setInputImage(pi);
								rgbtogray.process();
								images[i] = new JIUIntegerImage_Image3PixelAccess_Adaptor((IntegerImage)rgbtogray.getOutputImage());
							}
							else if (pi instanceof IntegerImage)
								images[i] = new JIUIntegerImage_Image3PixelAccess_Adaptor((IntegerImage)pi);
							else
								throw new AssertionError("Loaded image is not IntegerImage??? FATAL\n");
						}
						catch (Exception e)
						{
							System.out.println("Exception when reading "+imagePaths[i]);
							e.printStackTrace();
						}
					}
					else
						throw new AssertionError(String.format("Unable to load with data dimension of %d\n",datadim));
						
					if ( (w != -1) && ((images[i].getWidth() != w) || (images[i].getHeight() != h) || (images[i].getDepth() != d) ))
						System.err.printf("band image %s of different dimensions than labels image??? %d,%d,%d  %d,%d,%d\n",
								imagePaths[i],images[i].getWidth(),images[i].getHeight(), images[i].getDepth(), w,h,d);
				}
				w = images[0].getWidth();
				h = images[0].getHeight();
				d = images[0].getDepth();
			}
		}
		
		public void setRange(float imin, float imax)
		{
			this.IRangeMin = imin;
			this.IRangeMax = imax;
			this.IRangeScale = 1.0f / (IRangeMax - IRangeMin);
		}

		public void unloadData()
		{
			loaded = false;
			for (int i=0;i<images.length;i++)
				images[i] = null;
		}
		
	}
	static public class LikelihoodMapSet
	{
		Image3PixelAccess maps[];
		int classNum=2;
		String fileString;
		
		static private enum ARGTYPE
		{
			STRINGFMT,   // this is the original format -- like maps%02d.img
			MULTILINE    // this is added to support pipeline usage (a file containing the files!)
		}
		
		static private ARGTYPE checkArgType(String a)
		{
			int lio = a.lastIndexOf("%");
			if (lio == -1)
				return ARGTYPE.MULTILINE;
			else
				return ARGTYPE.STRINGFMT;
		}
		
		/**
		 * @param n The data item number in the Tu-lonidata set
		 */
		public LikelihoodMapSet(String mapfmt, int classNum, int dim)
		{
			if ((dim != 2) && (dim != 3))
				throw new AssertionError("LikelihoodMapSet cannot initalization from dim "+dim);
			
			this.fileString = mapfmt;
			this.classNum = classNum;
			
			System.out.printf("loading likelihood maps for appearance from disk\n");
			maps = new Image3PixelAccess[classNum];
			
			ARGTYPE atype = checkArgType(mapfmt);
			if (checkArgType(mapfmt) == ARGTYPE.MULTILINE)
			{
				BufferedReader in;
				String line;
				try 
				{
					int i=0;
					in = new BufferedReader( new FileReader(mapfmt) );
					line = in.readLine();
					while (line != null)
					{
						maps[i++] = load(line,dim);
						line = in.readLine();
					}
					in.close();
				} 
				catch (FileNotFoundException e) 
				{
					e.printStackTrace();
				} 
				catch (IOException e) 
				{
					e.printStackTrace();
				}
			}
			else
			{   // load the maps from a FMT string
				for (int i=0;i<classNum;i++)
				{
					maps[i] = load(String.format(fileString,i),dim);
				}
			}
			
			System.out.printf("loading likelihood maps done\n");
		}
		
		private Image3PixelAccess load(String fn,int dim)
		{
			Image3PixelAccess loadedMap=null;
			if (dim == 3)
			{
				AnalyzeHeader hdr = new AnalyzeHeader(fn);
				loadedMap = hdr.loadFromDisk();
			}
			else
			{
				try
				{
					PixelImage pi = ImageLoader.load(fn);
					if (pi instanceof IntegerImage)
						loadedMap = new JIUIntegerImage_Image3PixelAccess_Adaptor((IntegerImage)pi);
					else
						throw new AssertionError("Loaded LMS image is not IntegerImage??? FATAL\n");
				}
				catch (Exception e)
				{
					System.out.println("Exception when reading "+fn);
					e.printStackTrace();
				}
			}
			return loadedMap;
		}
	}
	
	public static final long version = 20070712;
	
	
	
	static int numClasses = -1;

	static PrintStream o=System.out;
	
	private static final long HIWORD = 0xFFFFFFFF00000000L;
	
	private static final long LOWORD = 0x00000000FFFFFFFFL;
	
	private static final SingleFloatArrayDataItem Tsfadi = new SingleFloatArrayDataItem();
	
	private static  int minSpawnLevel=1;
	
	/**
	 * Check the argument to see if we can determine the file type of the file.
	 * The default type that we will return is Multiline.  
	 * To check if the file is directly the image, we will first look at the extension.
	 * If the extension is .img,.hdr,.png,.bmp,.gif,.pgm,.ppm, it is treated as a direct image.
	 * Then, we check if appending .img to the string will locate a file (for Analyze).  also direct image.
	 * 
	 * If not a direct image, simply open the file get a line and see if there is a comma there
	 *  (we make the restriction that no filename can have commas)
	 * @param arg
	 * @return
	 */
	static private ARG0TYPE checkImageFileType(String arg)
	{
		int lio = arg.lastIndexOf('.');
		
		if (lio != -1)
		{
			String suffix = arg.substring(lio+1,arg.length());
			System.out.printf("parsing suffix %s\n",suffix);
			
			if ( (suffix.equalsIgnoreCase("HDR")) || 
			     (suffix.equalsIgnoreCase("IMG")) ||
			     (suffix.equalsIgnoreCase("BMP")) ||
			     (suffix.equalsIgnoreCase("GIF")) ||
			     (suffix.equalsIgnoreCase("PNG")) ||
			     (suffix.equalsIgnoreCase("PGM")) ||
			     (suffix.equalsIgnoreCase("TIF")) ||
			     (suffix.equalsIgnoreCase("PPM")) 
			   )
			   return ARG0TYPE.JUSTIMAGE;
		}
		
		// try to append .img or .hdr to the string 
		File f = new File (arg + ".img");
		if (f.exists())
			return ARG0TYPE.JUSTIMAGE;
		f = new File (arg + ".hdr");
		if (f.exists())
			return ARG0TYPE.JUSTIMAGE;
		
		// next we open the file and get the first line and check if it has a comma
		try 
		{
			BufferedReader in = new BufferedReader( new FileReader(arg) );
			String line = in.readLine(); 
			in.close();
			System.out.println("read the line "+line);
			
			lio = line.lastIndexOf(',');
			System.out.println("lio is "+lio);
			
			if (lio != -1)
				return ARG0TYPE.COMMASEP;
		} 
		catch (FileNotFoundException e) 
		{
			e.printStackTrace();
		} 
		catch (IOException e) 
		{
			e.printStackTrace();
		}
		
		return ARG0TYPE.MULTILINE;
	}
	/**
	 * @param args
	 */
	public static void main(String[] args)
	{
		PrintStream o = System.out;
		GraphShifts.o = o;
		
		o.printf(" -- starting up\n");
		
		if ( (args.length != 4) && (args.length != 5))
		{
			String usage = 
		"\n// usage"+
		"\n//  NOTE:  "+
		"\n//     (1) All 3D images should be in the Analyze file format.  When specifying paths to images"+
		"\n//          you can either use the prefix, prefix.hdr, or prefix.img.  All are acceptable, but "+
		"\n//          for simplicity, I recommend just using the prefix."+
		"\n//         For 2D images, most popular formats are okay EXCEPT for JPG."+
		"\n//         3D is the default dimension."+
		"\n//     (2) Commas (',') or percents ('%') are not allowed to appear in any filename."+
		"\n"+
		"\n//  args[0] is the image data.  You can specify it in three ways:"+
		"\n//     (1) Just put the image filename here.  (e.g., /data/images/pat0.t1w.img)" +
		"\n//           This method only works when you have a single image channel (or RGB natural image)"+
		"\n//     (2) Use a text file that points to the image file(s) to use (multi-channels supported)"+
		"\n//            Each line of the file specifies an image band/sequence,"+
		"\n//             with the first line specifying the manual labels."+
		"\n//            During segmentation, the manual labels are completely disregarded and only used for evaluation."+
		"\n//            Example:  Line 1:   /data/CCB40/images/pat0.labels"+
		"\n//                      Line 2:   /data/CCB40/images/pat0.t1w"+
		"\n//            Give the path to the text file as the argument."+
		"\n//     (3) Also use a text file that points to the image file(s) with a different format."+ 
		"\n//            Give each image file (again with the labels image first) separated by commas"+
		"\n//            Example:  Line 1:   /data/CCB40/images/pat0.labels,/data/CCB40/images/pat0.t1w"+
		"\n"+
		"\n//  args[1] is the likelihood map data.  You can specify it in two ways:"+
		"\n//     (1) Use a printf style string format to the likelihood maps (e.g., /data/map_%02d)"+
		"\n//     (2) Use a text file that point to the maps file -- sequentially, one path on each line."+
		"\n//            Example:  Line 1:   /data/map_00"+
		"\n//                      Line 2:   /data/map_01"+
		"\n//  args[2] number of classes (and thus likelihood maps) [integer]"+
		"\n//  args[3] path for writing out the final labels volume (path to write the output file)"+
		"\n"+
		"\n//  args[4] (optional) option flag string that specifies various parameters to control the graph-shifts "+
		"\n//           energy minimization process.  Format is flag1=value1,flag2=value2,etc... "+
		"\n"+
		"\n//          Allowable flags and default values: \n"+
		"\n"+
		"\n//          D=# (integer, default=3, range=2 or 3) -- Specifies dimension of the images."+  
		"\n//          L=# (integer,default=-1) -- Specifies how many shift levels to consider during the "+
		"\n//             energy minimization process.  -1 means consider all of the levels and it is default."+
		"\n//          M=# (integer,default=100000, range M>0) -- Maximum number of shifts to consider before stopping."+
		"\n//          W=# (float,default=0.7,range 0<=W<=1) -- Weight between appearance term and smoothness term.  "+
		"\n//               For appearance (A) and smoothness (S), the energy is weighted    E = W*A+(1-W)*S."+
		"\n//          T=# (integer,default=1,range {1,2}) Type of shifts to include.  For value 1, only the original "+
		"\n//               split/merge shifts are considered, for value 2, the split/merge and spawning shifts are "+
		"\n//               considered.  Including the spawning shifts increases the computational cost and should "+
		"\n//               only be used when necessary."+
		"\n//          V=# (integer,default=0, range >=0) Verbose output during shifting.  Outputs information every #-shifts."+
		"\n//               0 means no verbose output is made!"+
		"\n//          S=# (integer, default=2, range >= 0 to H) Min level at which to consider spawning (if T=2)."+
		"\n//          P=# (double, default=0.8 range = 0<x<=1) Appearance likelihood threshold when creating "+
		"\n//                spawn edges.  A spawn edge is created if assigned node likelihood is lower than this."+
		"\n"+
		"\n//          ********* Specifying alternate binary energy terms "+
		"\n"+
		"\n//          The default binary energy term is the boundary smoothness, which simply gives "+
		"\n//           a weight of 1 to the binary term between any bordering pixels having different models."+
		"\n//           You can specify alternate binary energy terms by giving the following parameter:"+
		"\n//          BEC=path/to/file  where path/to/file specifies the text file that contains the "+
		"\n//           information to define the binary energy computer."+
		"\n//          The BEC file is an ASCII file wherein the first line contains the type of binary term"+
		"\n//           and the following lines contains construction information."+
		"\n//          Types of terms and definitions:"+
		"\n//            context     -- second line contains number of classes, N, (integer) and the third through"+
		"\n//                            N+3 lines contain the energy terms defining the context matrix."+
		"\n//                           Each line is a row of the matrix with each entry separated by whitespace."+
		"\n"+
		"\n//          ********* Less Important And Less Useful Flags (that very seldom if ever get changed "+
		"\n"+
		"\n//          H=# (integer,default=4,range H>1) Maximum height of the hierarchical representation."+
		"\n//          R=# (float, default=0.15, range 0<R<1) Desired reduction factor in the initial bottom-up "+
		"\n//               construction of the graph hierarchy."+
		"\n"+
		"\n//          ********* For evaluation and testing robustness"+
		"\n"+
		"\n//          BAD=#:# (double,long, no defaults for values, but default is OFF)  This initiates a sequence of "+
		"\n//                  \"bad\" shifts that perturb the initialization to make the minimization more difficult."+
		"\n//                  The first is the scale-factor on the energy to achieve before stopping, the second is the random seed."+
		"\n//          MO=S:S (String to colormap file : String to write output image). Map-Output: This will take the"+
		"\n//                  final output labels and map them into RGB using the colormap file.  "+
		"\n//                  Currently only supported when D=2."+
		"\n";
			System.err.println(usage);
			System.exit(-1);
		}
		
		int opt_shiftLevels = -1;
		int opt_maxShifts = 100000;
		double opt_lambda = 0.7;
		int opt_shifts_type = 1;
		int opt_hierarchy_height=4;
		double opt_reduction=0.15;
		int opt_verbose = 0;
		int opt_minspawnlevel = 2;
		double opt_prunespawn = 0.1;
		boolean opt_dobad=false;
		double opt_badfactor=1;
		long opt_badseed=1;
		int opt_dim=3;
		boolean opt_bec=false;
		String opt_becpath=null;
		boolean opt_mapout=false;
		String opt_mapcmap=null;
		String opt_mapfn=null;
		
		if (args.length == 5)
		{
			String[] opts = args[4].split(",");
			for (String opt : opts)
			{
				String vals[] = opt.split("=");
				if (vals[0].equals("L"))
				{
					opt_shiftLevels = Integer.parseInt(vals[1]);
				}
				else  if (vals[0].equals("D"))
				{
					opt_dim = Integer.parseInt(vals[1]);
					if ((opt_dim != 2) && (opt_dim != 3))
					{
						System.err.printf("Bad value (%d) for dimension, setting to default 3\n",opt_dim);
						opt_dim = 3;
					}
				}
				else  if (vals[0].equals("M"))
				{
					opt_maxShifts = Integer.parseInt(vals[1]);
					if (opt_maxShifts < 1)
					{
						System.err.printf("Bad value (%d) for max shifts option, setting to default 100000\n",opt_maxShifts);
						opt_maxShifts = 100000;
					}
				}
				else  if (vals[0].equals("W"))
				{
					opt_lambda = Double.parseDouble(vals[1]);
					if ( (opt_lambda < 0) || (opt_lambda > 1))
					{
						System.err.printf("Bad value (%f) for energy term weight, setting to default 0.7\n",opt_lambda);
						opt_lambda = 0.7;
					}
				}
				else  if (vals[0].equals("T"))
				{
					opt_shifts_type = Integer.parseInt(vals[1]);
					if ((opt_shifts_type != 1) && (opt_shifts_type != 2))
					{
						System.err.printf("Bad value (%d) for shifts type option, setting to default 1\n",opt_shifts_type);
						opt_shifts_type = 1;
					}
				}
				else  if (vals[0].equals("H"))
				{
					opt_hierarchy_height = Integer.parseInt(vals[1]);
					if (opt_hierarchy_height <= 1)
					{
						System.err.printf("Bad value (%d) for max hierarchy height, setting to default 4\n",opt_hierarchy_height);
						opt_hierarchy_height = 4;
					}
				}
				else  if (vals[0].equals("R"))
				{
					opt_reduction = Double.parseDouble(vals[1]);
					if (opt_reduction > 1)
					{
						System.err.printf("Bad value (%f) for bottom-up reduction factor, setting to default 0.15\n",opt_reduction);
						opt_reduction = 0.15;
					}
				}
				else  if (vals[0].equals("V"))
				{
					opt_verbose = Integer.parseInt(vals[1]);
					if (opt_verbose < 0)
					{
						System.err.printf("Bad value (%d) for verbose output, setting to default 0\n",opt_verbose);
						opt_verbose = 0;
					}
				}
				else  if (vals[0].equals("S"))
				{
					opt_minspawnlevel = Integer.parseInt(vals[1]);
					if (opt_minspawnlevel < 0)
					{
						System.err.printf("Bad value (%d) for minspawnlevel output, setting to default 2\n",opt_minspawnlevel);
						opt_minspawnlevel = 2;
					}
				}
				else  if (vals[0].equals("P"))
				{
					opt_prunespawn = Double.parseDouble(vals[1]);
					if ( (opt_prunespawn <= 0) || (opt_prunespawn > 1)) 
					{
						System.err.printf("Bad value (%f) for prune spawn threshold, setting to default 0.1\n",opt_prunespawn);
						opt_prunespawn= 0.1;
					}
				}
				else  if (vals[0].equals("BAD"))
				{
					opt_dobad = true;
					String vals2[] = vals[1].split(":");
					opt_badfactor = Double.parseDouble(vals2[0]);
					opt_badseed = Long.parseLong(vals2[1]);
				}
				else  if (vals[0].equals("BEC"))
				{
					opt_bec = true;
					opt_becpath = vals[1];
				}
				else  if (vals[0].equals("MO"))
				{
					opt_mapout = true;
					String vals2[] = vals[1].split(":");
					opt_mapcmap = vals2[0];
					opt_mapfn = vals2[1];
				}
				else
				{
					System.err.printf("Skipping misunderstood parameter string: %s.\n",opt);
				}
			}
		}
		
		System.out.printf("Running GraphShifts code version %d\n",version);
		
		
		String lmapfmt = args[1];
		int classNum = Integer.parseInt(args[2]);
		GraphShifts.numClasses = classNum;
		
		o.printf("reading image data\n");
		ARG0TYPE arg0type = checkImageFileType(args[0]);
		System.out.println("arg0 is type "+arg0type);
		
		ImageData ID;
		if (arg0type == ARG0TYPE.MULTILINE)
			ID = ImageData.createFromMLFile(args[0],opt_dim);
		else if (arg0type == ARG0TYPE.COMMASEP)
			ID = ImageData.createFromCSFile(args[0],opt_dim);
		else
			ID = ImageData.createFromImage(args[0],opt_dim);
			
		ID.loadData();
		
		LikelihoodMapSet lms = new LikelihoodMapSet(lmapfmt,classNum,opt_dim);
		
		o.printf("working with %d classes\n",classNum);
		o.printf("data dimension is %d\n",opt_dim);
		o.printf("working with %d shift levels with %d maximum shifts \n",opt_shiftLevels,opt_maxShifts);
		o.printf("doing graph-shifts type %d\n",opt_shifts_type);
	
		o.printf("initializing the object\n");
		GraphShifts a = new GraphShifts(ID,lms);
		
		
		if (opt_bec)
		{
			o.printf("updating binary energy computer:\n",opt_becpath);
			try
			{
				BufferedReader becbr = new BufferedReader(new FileReader(opt_becpath));
				
				String bectype = becbr.readLine();
				
				if (bectype.equalsIgnoreCase("context"))
				{
					System.out.printf("BEC: using a context binary energy computer\n");
					a.BEC = new BECContext(becbr);
				}
				else
				{
					System.err.println("Do not understand BEC of type: "+bectype+"  continuing with std smoothness");
				}
			
				becbr.close();
			}
			catch(IOException e)
			{
				a.BEC = new BECSmoothness(); 
				System.err.printf("Binary Energy Term creation from user specificied path failed.\n");
				System.err.printf("This is not fatal-will just use standard smoothness term.\n");
				e.printStackTrace();
			}
			
		}
		
		
		o.printf("initialization observers\n");
		
//		ShiftSequenceMaker sequenceMaker = 
//			new ShiftSequenceMaker(image,115,1.0f,"/tmp/shifts/shift%05d.png",lms.classNum,500);
		
//		ShiftStatistician statistician = 
//			new ShiftStatistician("/Users/jcorso/Desktop/stats.txt");
		
//		ShiftTotalEnergy teObserver = 
//			new ShiftTotalEnergy("/Users/jcorso/Desktop/energy.txt",a);
		
//		ShiftStartEndTotalEnergy setObserver = 
//			new ShiftStartEndTotalEnergy("/Users/jcorso/Desktop/energyse.txt",a);
		
//		ShiftWeightAccumulator waObserver = 
//			new ShiftWeightAccumulator("/Users/jcorso/Desktop/weight.txt",a);
//		
//		ShiftProcessVoyeur pvObserver = 
//			new ShiftProcessVoyeur(ID.getImage(0),"/data/lotushill/lhi-12.lut",
//					"/data/lotushill/iccv2007/tempdir/process","-1","fxy,0","0,1,2,3,4,5,6,7,8,9,10,11");
//					"/data/lotushill/iccv2007/tempdir/process","1,5,10,25,50,100,500,1000,1500,2000,2500,3000,3500,4000,4500,5000","fxy,0","0,1,2,3,4,5,6,7,8,9,10,11");
//		  for image 00, the slices we want are "xy,116,xz,57,zy,39"
		
//		SCShiftProcessVoyeur3D j3dObserver = 
//			new SCShiftProcessVoyeur3D(image,"/data/Tu-lonidata/labels-fixed/subcortical.lut",
//					"/Users/jcorso/Desktop/outlines/process00","-1","1,2");
		
//		a.addObserver(sequenceMaker);
//		a.addObserver(statistician);
//		a.addObserver(teObserver);
//		a.addObserver(waObserver);
//		a.addObserver(setObserver);
//		a.addObserver(pvObserver);
//		a.addObserver(j3dObserver);
		
		a.lambdaAppearance = opt_lambda;
		a.lambdaSmooth = 1-opt_lambda;
		o.printf("working with appearance coeff %f and smoothness coeff %f\n",a.lambdaAppearance,a.lambdaSmooth);
		
		if (opt_verbose > 0)
		{
			a.verboseB = true;
			a.verboseF = opt_verbose;
		}
		else
		{
			a.verboseB = false;
		}
		
		a.minSpawnLevel = opt_minspawnlevel;
		
		
		Date start,stop,startI,stopI;
		
		startI = new Date();
		// normal run is 4,5... set to 0 for only voxel layer and top-layer
		a.initializeGraph(new EdgeSampleInitializer(lms,opt_hierarchy_height,opt_reduction,0.1,minSpawnLevel),(opt_shifts_type==2),opt_prunespawn);
		
		stopI = new Date();
		
		if (opt_dobad)
			a.doBadGraphShifts(opt_badfactor,opt_badseed);
		
		
		start = new Date();
		
		a.doGraphShifts(opt_shiftLevels,opt_maxShifts);
		
		stop = new Date();

		// output some statistics too
		int seconds,minutes;
		seconds = (int)( (stopI.getTime()-startI.getTime()) / 1000 );
		minutes = seconds / 60;
		seconds = seconds % 60;
		System.out.printf("initialization took %d:%02d (m:s)\n",minutes,seconds);
		seconds = (int)( (stop.getTime()-start.getTime()) / 1000 );
		minutes = seconds / 60;
		seconds = seconds % 60;
		System.out.printf("graph shifts took %d:%02d (m:s)\n",minutes,seconds);
		seconds = (int)( (stop.getTime()-startI.getTime()) / 1000 );
		minutes = seconds / 60;
		seconds = seconds % 60;
		System.out.printf("total took %d:%02d (m:s)\n",minutes,seconds);
		
//		o.printf("writing the final hierarchy to disk\n");
//		ObjectToDisk.save("/tmp/hierarchy.object", a.hierarchy);
		
		o.printf("writing the final labels image to disk\n");
		ScalarImageB3 labels = a.getFinalLabels();
		if (opt_dim == 3)
		{
			AnalyzeHeader hdr = new AnalyzeHeader(args[3],labels);
			hdr.saveToDisk();
		}
		else if (opt_dim == 2)
		{
			ImageUtilities.writeViaJIU(labels, args[3], Image3Protocol.SLICE_XY, 0);
		}
		o.printf("done.\n");
		
		if (opt_mapout)
		{
			if (opt_dim != 2)
				System.out.printf("Unsupported colormapping in 3D right now");
			o.printf("mapping final labels into rgb with %s\n",opt_mapcmap);
			Colormap cmap = Colormap.createFromFile(opt_mapcmap);
			int w = labels.getWidth();
			int h = labels.getHeight();
			MemoryRGB24Image RGB = new MemoryRGB24Image(w,h);
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
				{
					int packed = cmap.getColorAsIntARGB( labels.getPixelInt(x, y, 0)  );
					RGB.putSample(MemoryRGB24Image.INDEX_RED,x,y,   (0x00FF0000 & packed) >> 16);
					RGB.putSample(MemoryRGB24Image.INDEX_GREEN,x,y, (0x0000FF00 & packed) >> 8);
					RGB.putSample(MemoryRGB24Image.INDEX_BLUE,x,y,  (0x000000FF & packed) );	
				}
			
			ImageUtilities.writeViaJIU(RGB, opt_mapfn);
		}
	}
	
	
	
	/*********  Beginning of member variables  **********/
	
	double lambdaSmooth = 0.50;
	double lambdaAppearance = 0.50;
	HardGraphHierarchy hierarchy;
	ThinGraph tgHierarchy[];

	ImageData image;
	
	LikelihoodMapSet lms;
	double[] LLmapLUT;  // store Log(0..255)  so we need not call log for the values 
	TLongHashSet shifts;
	
	int w,h,d,wh;
	private int unpackedLevel;
	private int unpackedNodeId;
	private int updateShiftsDownCount;
	
	private ArrayList<GraphShiftsObserver> observers;
	private boolean haveObservers=false;
	
	TLongHashSet shiftsTouched;
	
	
	//	Each shift that will spawn a new root-level node has an entry in this hash-map.  The 
	//  "value" of this entry gives the index of the class.
	TLongByteHashMap spawnShifts;

	int computeShiftCounter=0;
	
	// this guy is used whenever an int hash is required.  but initializing it locally each time would be
	//  a waste...
	private TIntHashSet tempIntHashSet = new TIntHashSet();

	private TIntIntHashMap tempIIHM = new TIntIntHashMap(10);
	
	TIntIntHashMap boundaryLengthIIHM;
	
	TIntHashSet newNgbFromChildIHS;
	
	float LBUF[] = new float[numClasses];
	
	//	 governs verbose output during shifting
	public boolean verboseB = false;
	public int verboseF = 0;
	
	FixedModelBinaryEnergyComputer BEC = new BECSmoothness();
	
	/**** Constructor ******/
	
	public GraphShifts(ImageData image, LikelihoodMapSet lms)
	{
		this.image = image;
		w = image.getWidth();
		h = image.getHeight();
		d = image.getDepth();
		wh = w*h;
		
		this.lms = lms;
		// instantiate the data structure to store the graph shifts.  the initial capacity is not too important
		shifts = new TLongHashSet(image.getWidth());
		spawnShifts = new TLongByteHashMap();
		
		boundaryLengthIIHM = new TIntIntHashMap(20);
		newNgbFromChildIHS = new TIntHashSet(10);
		
		observers = new ArrayList<GraphShiftsObserver>(1);
		
		LLmapLUT = new double[256];
		LLmapLUT[0] = -7.0f;
		for (int i=1;i<256;i++)
		{
			double v = (double)i / 255.0f;
			LLmapLUT[i] = Math.log(v);
		}
		
		o.printf("%s Class:  done construction\n",this.getClass().getSimpleName());
	}
	
	public void addObserver(GraphShiftsObserver O)
	{
		observers.add(O);
		haveObservers=true;
	}
	
	
	/*** ----------- Debug Code to Check Boundary Lengths ---------- ***/
	public void checkBoundaryLengths()
	{
		for (int layer=tgHierarchy.length-2;layer>0;layer--)
		{
			ThinGraph qG = tgHierarchy[layer];
			for (int qi=0;qi<qG.getNumberOfNodes();qi++)
			{
				int qn = qG.NDgetNumberOfNeighbors(qi);
				for (int qni=0;qni<qn;qni++)
				{
					int qnid = qG.NDgetNeighbor(qi,qni);
					int uBL = computeBoundaryLengthToNeighborViaChildren(layer,qi,qnid);
					int nBL = qG.NDgetBoundaryLength(qi,qni);
					if (uBL != nBL)
						o.printf("layer %d check %d, n#%d (id%d), uBL is %d and nBL is %d\t\t\t%b\n",layer,qi,qni,qnid,uBL,nBL,(uBL==nBL));
				}
			}
		}
	}
	
	
	public void checkBoundaryLengths(int layer)
	{
		ThinGraph qG = tgHierarchy[layer];
		for (int qi=0;qi<qG.getNumberOfNodes();qi++)
		{
			int qn = qG.NDgetNumberOfNeighbors(qi);
			for (int qni=0;qni<qn;qni++)
			{
				int qnid = qG.NDgetNeighbor(qi,qni);
				int uBL = computeBoundaryLengthToNeighborViaChildren(layer,qi,qnid);
				int nBL = qG.NDgetBoundaryLength(qi,qni);
				o.printf("check %d, n#%d (id%d), uBL is %d and nBL is %d\t\t\t%b\n", qi,qni,qnid,uBL,nBL,(uBL==nBL));
			}
		}
	}
	
	public int computeBoundaryLengthToNeighborViaChildren(int level, int nodeIndex, int neighborIndex)
	{
		int d=0;
		
		int nc = tgHierarchy[level].NDgetNumberOfChildren(nodeIndex);
		for (int ci=0;ci<nc;ci++)
		{
			int cid = tgHierarchy[level].NDgetChild(nodeIndex, ci);
			
			int n = tgHierarchy[level-1].NDgetNumberOfNeighbors(cid);
			for (int i=0;i<n;i++)
			{
				int ni = tgHierarchy[level-1].NDgetNeighbor(cid, i);
				if (tgHierarchy[level-1].NDgetParent(ni) == neighborIndex)
					d += tgHierarchy[level-1].NDgetBoundaryLength(cid, i);
			}
		}
		
		return d;
	}
	
	
	
	/** 
	 * This computes the log-likelihood of the appearance model given the class C.
	 * To turn it into an "energy" take negative of the returned value.
	 * @param level
	 * @param nodeIndex
	 * @param C
	 * @return
	 */
	public double computeLLAtLeaves(int level, int nodeIndex, Image3PixelAccess LLmap)
	{
		ThinGraph G = tgHierarchy[level];
		int nc = G.NDgetNumberOfChildren(nodeIndex);
		if (nc == 0)
		{ 
			if (level != 0)
				return 0;  // this guy should not compute anything! he's not a leaf
			
			int x,y,z;
			z = ((int)((double)nodeIndex / (double)(wh)));
			y = ((int)((double)(nodeIndex-z*wh) / (double)w));
			x = ((nodeIndex-z*wh) % w);
//			float v = LLmap.getPixelFloat(x, y, z);
//			return (v==0.0f) ? -100.0f : Math.log(v/255.0f);
			return LLmapLUT[LLmap.getPixelInt(x,y,z)];
			// this just assigns a convention that log(0) is a very "large" negative (relatively, speaking)
		}
		
		double LL = 0;
		for (int i=0;i<nc;i++)
			LL += computeLLAtLeaves(level-1,G.NDgetChild(nodeIndex,i),LLmap);
		return LL;
	}
	
	public double computeLLAtLeavesCache(int level, int nodeIndex, int cIndex)
	{
		ThinGraph G = tgHierarchy[level];
		
		// if the level is 1, then we use special case and go to the llmap
		if (level == 0)
		{
			Image3PixelAccess LLmap = lms.maps[cIndex];
			int x,y,z;
			z = ((int)((double)nodeIndex/ (double)(wh)));
			y = ((int)((double)(nodeIndex-z*wh) / (double)w));
			x = ((nodeIndex-z*wh) % w);
			return LLmapLUT[LLmap.getPixelInt(x,y,z)];
		}
		
		if (level == 1)
		{ 
			int nc = G.NDgetNumberOfChildren(nodeIndex);
			if (nc == 0)
				return 0;  // this guy should not compute anything! he's not a leaf
			Image3PixelAccess LLmap = lms.maps[cIndex];
			
			// otherwise, go through each child and accumulate from the LLmapLut
			double d=0;
			for (int ni=0;ni<nc;ni++)
			{
				int cid = G.NDgetChild(nodeIndex, ni);
				int x,y,z;
				z = ((int)((double)cid/ (double)(wh)));
				y = ((int)((double)(cid-z*wh) / (double)w));
				x = ((cid-z*wh) % w);
				d += LLmapLUT[LLmap.getPixelInt(x,y,z)];
			}
			return d;
		}
		
		// otherwise, just go through the children and get the likelihoods from their storage
		ThinGraph G_ = tgHierarchy[level-1];
		int nc = G.NDgetNumberOfChildren(nodeIndex);
		if (nc==0)
			return 0;  // this guy should not compute anything! he's not a leaf
		double LL = 0;
		for (int i=0;i<nc;i++)
		{
			int cid = G.NDgetChild(nodeIndex, i);
			LL += G_.NDgetLikelihood(cid, cIndex);
		}
		return LL;
	}
	
	
	/**
	 * Computes the boundary length of the nodeIndex to all neighbors not of the classIndex
	 * @param level
	 * @param nodeIndex
	 * @param classIndex
	 * @return
	 */
	public int computeMultiBoundaryLength(int level, int nodeIndex, int classIndex)
	{
		int d=0;
		
		int n = tgHierarchy[level].NDgetNumberOfNeighbors(nodeIndex);
		for (int i=0;i<n;i++)
		{
			int ni = tgHierarchy[level].NDgetNeighbor(nodeIndex, i);
			if (tgHierarchy[level].NDgetClassIndex(ni) != classIndex)
				d += tgHierarchy[level].NDgetBoundaryLength(nodeIndex, i);
		}
		
		return d;
	}
	
	public double computeSegmentationLL()
	{
		ThinGraph top = tgHierarchy[tgHierarchy.length-1];
		int topNum = top.getNumberOfNodes();
		double LL=0;
		double dE,bE;
		
		for (int i=0;i<topNum;i++)
		{
			dE = -computeLLAtLeaves(tgHierarchy.length-1,i,lms.maps[top.NDgetClassIndex(i)]);
			bE = 0.5 * computeTopLevelBoundaryLength(i);
			
			LL += lambdaAppearance*dE + lambdaSmooth*bE;
		}
		
		return LL;
	}
	
	
	/**
	 * This function computes the best shift for node uId on level level in the graph.   The function will 
	 *  return the id of the shift and store the id and weight of the shift in the actual graph.  The function
	 *  does not query the shift hashset and neither does it return the packed shift because the caller can easily
	 *  pack it with the level and uId given as parameters.
	 *  If there is no best shift (e.g. all neighbors share same model), then -1 is returned
	 *  and -1,0 is stored in the graph.
	 * 
	 * As the function will exhaustively evaluate all neighbors of the given uId node, it completely disregards 
	 *  the current shift stored at uId.
	 * 
	 * @param level
	 * @param uId
	 * @return -1 If there is no shift to take that is going to reduce the energy
	 * @return -2 If there is no shift to take at all (all neighbors share same class)
	 */
	private int computeShift(int level, int uId)
	{
		computeShiftCounter++;
//		o.printf("computing shift at level %d id %d, count %d\n",level,uId,computeShiftCounter);
		
			
		// this is for debugging and wasted computation in production code.
//		if ( (level >= minSpawnLevel) &&  (uId == 0) )
//			throw new AssertionError ("Computing shift from a spawn node...\n");
		
		// we begin by removing the shift for level, uid from the spawn set (if it is in there)
		spawnShifts.remove(packShift(level, uId));
		
		int uCI = tgHierarchy[level].NDgetClassIndex(uId); 
		int unN = tgHierarchy[level].NDgetNumberOfNeighbors(uId);
		double bestDelta = -0.00001;
		int bestVId = -1;
		boolean attempted=false;
		boolean bestShiftIsSpawn = false;
		
		double uLL = Double.POSITIVE_INFINITY;
		for (int uni=0;uni<unN;uni++)
		{
			int vId = tgHierarchy[level].NDgetNeighbor(uId, uni);
			
			if ( (vId == 0) && (level >= minSpawnLevel) )
			{
				// cl is the new class to switch to
				for (int cl=0;cl<numClasses;cl++)
				{
					if (cl==uCI)
						continue;
					
					attempted=true;
					
					if (uLL == Double.POSITIVE_INFINITY)
						uLL = -computeLLAtLeavesCache(level,uId,uCI);
					double vLL = -computeLLAtLeavesCache(level,uId,cl);
					double deltaAppearance = vLL - uLL;  //(u is current, being removed, v would be new being added)

					
					//  when considering only the boundary length terms:
					// the change is the newBL (connection to everything) minus the 
					//  old boundary length (connection to everything but its class uCI)
					// double newBL = computeTotalBoundaryLength(level,uId);
					// double oldBL = computeMultiBoundaryLength(level,uId,uCI);
					// So, the resulting change in boundary length from this guy is simply the 
					//  length to same-classed (uCI) nodes)
					//double deltaBL = computeSingleBoundaryLength(level,uId,uCI);
					// However, when considering an arbitrary fixed model boundary term, then
					//  we need to go through all of this guys neighbors to compute the new boundary contribution
					double deltaBL = 0;
					double binaryOld=0;
					double binaryNew=0;
					for (int un=0;un<unN;un++)
					{
						int cluni = tgHierarchy[level].NDgetNeighbor(uId, un);
						if (cluni == 0)
							continue;
						int cllength = tgHierarchy[level].NDgetBoundaryLength(uId, un);
						int clunc = tgHierarchy[level].NDgetClassIndex(cluni);
						
						binaryOld += BEC.computeEnergy(cllength, uCI, clunc);
						binaryNew += BEC.computeEnergy(cllength, cl, clunc);
					}
					deltaBL = binaryNew - binaryOld;
					
					double delta = lambdaAppearance * deltaAppearance + 
					               lambdaSmooth     * deltaBL;

					if (delta < bestDelta)
					{
						bestDelta = delta;
						bestVId = cl;
						bestShiftIsSpawn = true;
					}
				}
			}
			else
			{
				int vCI = tgHierarchy[level].NDgetClassIndex(vId);
				if (uCI != vCI)
				{
					attempted=true;

					// the shift we are evaluating is u --> v
					// really, we should be storing the current energy (or log-likelihood) at each node
//					if (uLL == Double.POSITIVE_INFINITY) // only want to calculate this once
//					uLL = -computeLLAtLeaves(level,uId,lms.maps[uCI]);
//					double vLL = -computeLLAtLeaves(level,uId,lms.maps[vCI]);
					if (uLL == Double.POSITIVE_INFINITY) // only want to calculate this once
						uLL = -computeLLAtLeavesCache(level,uId,uCI);
					double vLL = -computeLLAtLeavesCache(level,uId,vCI);
					double deltaAppearance = vLL - uLL;  //(u is current, being removed, v would be new being added)

					/*  This block of code is applicable only for a Potts/Ising model type of 
					 *  binary term.  It only consider neighbors of the current and new classes.
					 *  For general binary energies you have to use the next block of code.
					 *
					int lengthToNEW = computeSingleBoundaryLength(level, uId, vCI);
					int lengthToOLD = computeSingleBoundaryLength(level, uId, uCI);
					
					// these are somewhat unnecessarily costly because
					//  in most of the BEC's we'll use, the energy is 0 for same classes
					//  and the different classes are symmetric.  so, we can simply
					//  take the difference of lengthToNEW - lengthToOLD
					//  and then only need to call BEC.computeEnergy once.  but, let's keep 
					//  it this way for generality now.
					double binaryOld = BEC.computeEnergy(lengthToNEW, uCI, vCI) +
					                 + BEC.computeEnergy(lengthToOLD, uCI, uCI);
					double binaryNew = BEC.computeEnergy(lengthToNEW, vCI, vCI) +
					                 + BEC.computeEnergy(lengthToOLD, vCI, uCI);
					*/
					
					
					// we have to go through each of uId's neighbors and accumulate binary energies
					//  for uId's current class and its future one
					double binaryOld=0;
					double binaryNew=0;
					for (int un=0;un<unN;un++)
					{
						int cluni = tgHierarchy[level].NDgetNeighbor(uId, un);
						if (cluni == 0)
							continue;
						int cllength = tgHierarchy[level].NDgetBoundaryLength(uId, un);
						int clunc = tgHierarchy[level].NDgetClassIndex(cluni);
						binaryOld += BEC.computeEnergy(cllength, uCI, clunc);
						binaryNew += BEC.computeEnergy(cllength, vCI, clunc);
					}
					
					double deltaBinary = binaryNew - binaryOld;
					double delta = lambdaAppearance * deltaAppearance + 
							       lambdaSmooth     * deltaBinary;

					if (delta < bestDelta)
					{
						bestDelta = delta;
						bestVId = vId;
						bestShiftIsSpawn=false;
					}
				}
			}
		}
		
		// check if there is actually a shift to take, if not, return -1 and set shift to -1,0
		if (bestVId == -1)
		{ 
			tgHierarchy[level].NDsetGraphShift(uId, -1, 0);
			if (!attempted)
				return -2;
			return -1;
		}
		
		// we have a shift, set it in the graph and return the index of the node to which the shift is defined
		if (bestShiftIsSpawn)
		{
			tgHierarchy[level].NDsetGraphShift(uId, 0, (float)bestDelta);
			spawnShifts.put(packShift(level,uId), (byte) bestVId);
			return 0;
		}
		else
		{
			tgHierarchy[level].NDsetGraphShift(uId, bestVId, (float)bestDelta);
			return bestVId;
		}
	}

	
	/**
	 * Compute the length of the boundary for nodeIndex to all its neighbors that have a class
	 *  of classIndex.  This could be the same class index as the node itself.
	 * @param level
	 * @param nodeIndex
	 * @param classIndex
	 * @return
	 */
	public int computeSingleBoundaryLength(int level, int nodeIndex, int classIndex)
	{
		int d=0;
		
		int n = tgHierarchy[level].NDgetNumberOfNeighbors(nodeIndex);
		for (int i=0;i<n;i++)
		{
			int ni = tgHierarchy[level].NDgetNeighbor(nodeIndex, i);
			if (tgHierarchy[level].NDgetClassIndex(ni) == classIndex)
				d += tgHierarchy[level].NDgetBoundaryLength(nodeIndex, i);
		}
		
		return d;
	}

	
	/**
	 * Compute the gradient of the shift at level level from uId to neighbor index nid.
	 * @param level
	 * @param uId
	 * @param nid  neighbor index, not graph node index
	 * @return
	 */
	private double computeShiftSingle(int level, int uId, int nid)
	{
		int uCI = tgHierarchy[level].NDgetClassIndex(uId); 
		int vId = tgHierarchy[level].NDgetNeighbor(uId, nid);
		int vCI = tgHierarchy[level].NDgetClassIndex(vId); 
		
		double uLL = -computeLLAtLeavesCache(level,uId,uCI);
		double vLL = -computeLLAtLeavesCache(level,uId,vCI);
		double deltaAppearance = vLL - uLL;  //(u is current, being removed, v would be new being added)
				
		
		
		int lengthToNEW = computeSingleBoundaryLength(level, uId, vCI);
		int lengthToOLD = computeSingleBoundaryLength(level, uId, uCI);
		
		// these are somewhat unnecessarily costly because
		//  in most of the BEC's we'll use, the energy is 0 for same classes
		//  and the different classes are symmetric.  so, we can simply
		//  take the difference of lengthToNEW - lengthToOLD
		//  and then only need to call BEC.computeEnergy once.  but, let's keep 
		//  it this way for generality now.
		double binaryOld = BEC.computeEnergy(lengthToNEW, uCI, vCI) +
		                 + BEC.computeEnergy(lengthToOLD, uCI, uCI);
		double binaryNew = BEC.computeEnergy(lengthToNEW, vCI, vCI) +
		                 + BEC.computeEnergy(lengthToOLD, vCI, uCI);
		
		double deltaBinary = binaryNew - binaryOld;

		return lambdaAppearance * deltaAppearance + 
		       lambdaSmooth     * deltaBinary;
	}
	
	public int computeTopLevelBoundaryLength()
	{
		int d=0;
		int lv = hierarchy.getNumberOfLayers()-1;
		int n = tgHierarchy[lv].getNumberOfNodes();
		for (int i=0;i<n;i++)
			d += computeTopLevelBoundaryLength(i);
		return d;
	}

	/**
	 * Compute the boundary length for a top-level node.  Need to go to the children
	 *  because the toplevel nodes store no neighbors.  Computed to all of its neighbors
	 * @param level
	 * @param nodeIndex
	 * @return
	 */
	public int computeTopLevelBoundaryLength(int nodeIndex)
	{
		int d=0;
		int lv = hierarchy.getNumberOfLayers()-1;
		
		int cl = tgHierarchy[lv].NDgetClassIndex(nodeIndex);
		int nc = tgHierarchy[lv].NDgetNumberOfChildren(nodeIndex);
		for (int ci=0;ci<nc;ci++)
		{
			int cid = tgHierarchy[lv].NDgetChild(nodeIndex, ci);
			
			int ncn = tgHierarchy[lv-1].NDgetNumberOfNeighbors(cid);
			for (int cni=0;cni<ncn;cni++)
			{
				int cnid = tgHierarchy[lv-1].NDgetNeighbor(cid, cni);
				if (tgHierarchy[lv-1].NDgetClassIndex(cnid) != cl)
					d += tgHierarchy[lv-1].NDgetBoundaryLength(cid, cni);
			}
		}
		
		return d;
	}
	

	
	/**
	 * Compute the total boundary length of nodeIndex to all of its neighbors.  (Indifferent to 
	 *  the class of the neighbors)
	 * @param level
	 * @param nodeIndex
	 * @param classIndex
	 * @return
	 */
	public int computeTotalBoundaryLength(int level, int nodeIndex)
	{
		int d=0;
		
		int n = tgHierarchy[level].NDgetNumberOfNeighbors(nodeIndex);
		for (int i=0;i<n;i++)
			d += tgHierarchy[level].NDgetBoundaryLength(nodeIndex, i);
		
		return d;
	}
	
	
	/**  This function takes the hierarchy (initialization, assumed) and mucks around with it
	 * to harm the initialization.
	 * It randomly selects a shift and takes it...bad only.
	 * It still will only take a shift if it will actually change the energy function (between 
	 * two nodes with different models).
	 *
	 */
	public void doBadGraphShifts(double factor, long seed)
	{
		int hLevelNum = hierarchy.getNumberOfLayers();
		int shiftsTaken=0;
		
		Random R = new Random(seed);
		double totalGradient=0;
		
		double Estart = computeSegmentationLL();
		
		double Eneed = Estart * factor;
		
		while (totalGradient < Eneed)
		{
			// three random choices are made each iteration
			//  choose the level, choose the node and choose the neighbor to shift to
			int rL = R.nextInt(hLevelNum-1);
			ThinGraph G = tgHierarchy[rL];
			int nn = G.getNumberOfNodes();
			int rN = R.nextInt(nn);
			int sn = G.NDgetNumberOfNeighbors(rN);
			if (sn<=0)
				continue;
			int sni = R.nextInt(sn);
			int sN = G.NDgetNeighbor(rN, sni);
			
			// shift will be from rN to sN
			
			// check if the models are different
			int rM,sM;
			rM = G.NDgetClassIndex(rN);
			sM = G.NDgetClassIndex(sN);
			
			if (rM == sM)
				continue;
			
			double gradient = computeShiftSingle(rL,rN,sni);
			
			if (gradient <= 0)
				continue;
			
			totalGradient+=gradient;
			
			int rP,sP;
			rP = G.NDgetParent(rN);
			sP = G.NDgetParent(sN);
			
			System.out.printf("perturbation %d, level %d from %d to %d (parents %d to %d) with gradient %f\n",
					shiftsTaken,rL,rN,sN,rP,sP,gradient);
			
			pushClassIndexDown(rL,rN,sM);

			// change the parent of the node
			tgHierarchy[rL].NDsetParent(rN, sP);
			tgHierarchy[rL+1].NDaddChild(sP, rN);
			tgHierarchy[rL+1].NDremoveChildId(rP, rN);
			
			// the first block updates information in the hierarchy at and above the shift level
			if (rL < tgHierarchy.length-2)
			{
				updateBoundaryLengthsUpSmart(rL,rN,rP,sP);
				updateLikelihoodCachesUp(rL,rN,rP,sP);
			}
			
			shiftsTaken++;
		}
		System.out.printf("The total perturbation gradient is %f\n",totalGradient);
		double Eend = computeSegmentationLL();
		System.out.printf("Energy before perturbation %f and after %f\n",Estart,Eend);
		
//		ScalarImageB3 labels = getFinalLabels();
//		AnalyzeHeader hdr = new AnalyzeHeader("/tmp/perturbation",labels);
//		hdr.saveToDisk();
	}
	
	/**
	 * Do the graph shifts operation.
	 * 
	 * @param highLevel  what level should we consider to be the highest at which to compute the shifts?
	 *                   -1 means, use all possible layers.  Obviously, 0 is the smallest useful value.
	 * @param maxShiftNumber is the maximum number of shifts to take.
	 */
	public void doGraphShifts(int highLevel,int maxShiftNumber)
	{
		o.printf("%s Class:  computing the initial set of graph shifts\n",this.getClass().getSimpleName());
		int hLevelNum = hierarchy.getNumberOfLayers();
		int startLevel = (highLevel == -1) ? hLevelNum-2 : highLevel;
		
		int numPotentialShifts = 0;
		for (int level=startLevel;level>=0;level--)
		{
			ThinGraph G = tgHierarchy[level];
			int uN = G.getNumberOfNodes();
			
			for (int ui=(level >= minSpawnLevel)?1:0;ui<uN;ui++)
			{
				// this is only for debug code
				int unN = G.NDgetNumberOfNeighbors(ui);
				numPotentialShifts += unN;
				
				int vi = computeShift(level,ui);
				if (vi >= 0)
				{
					shifts.add(packShift(level, ui));
//					
//					float w = tgHierarchy[level].NDgetGraphShiftWeight(ui);
//					if ((level >= minSpawnLevel) && (vi == 0))
//					{
//						System.out.printf("[%d] spawn shift  lvl %d uid %d with weight %f\n",packShift(level,ui),level,ui,w);
//					}
//					else
//					{
//						System.out.printf("[%d] s/m   shift  lvl %d uid %d with weight %f\n",packShift(level,ui),level,ui,w);
//					}
				}
			}   // end of iterating nodes on level
		} 	// end of iterating levels 
		
		o.printf("%s Class:  computed %d initial shifts (%.3f%%)\n",this.getClass().getSimpleName(),
				shifts.size(),100.0f*(float)shifts.size()/(float)numPotentialShifts);
		o.printf("the number of potential shifts (edges) is %d\n",numPotentialShifts);
		
		// this guys stores what shifts have been updated each iteration so that we don't have to 
		//  check the same potential shift more than once
		shiftsTouched = new TLongHashSet(shifts.size());
		
		
		int shiftCount=0;
		
		PrintWriter pw = null;
		/*
		try {
			pw = new PrintWriter(new File("/tmp/shiftsLL.txt"));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		// */
		
		fireShiftBeginToObservers();
		
		while (shiftCount < maxShiftNumber)
		{
			/*
			double LL = computeSegmentationLL();
//			pw.printf("at shift %d, log-likelihood of segmentation is %f\n", shiftCount, LL);
			pw.printf("%f\n",LL);
			// */
			
			float bestShiftWeight = Float.POSITIVE_INFINITY;
			long  bestShift = -1;
			TLongIterator lit = shifts.iterator();
			while (lit.hasNext())
			{
				long ln = lit.next();
				float w = tgHierarchy[(int)((ln >> 32) & LOWORD)].NDgetGraphShiftWeight((int)(ln & LOWORD));
				if (w < bestShiftWeight)
				{
					bestShiftWeight = w;
					bestShift = ln;
				}
			}
			
			if (bestShiftWeight >= 0)
			{
				if (verboseB)
					o.printf("not proceeding with the graph shift because the best shift will decrease likelihood\n");
				break;
			}
			shiftCount++;

			unpackShift(bestShift);
			final int shiftLv = unpackedLevel;
			final int shiftLi = unpackedNodeId;
//			shiftLv = (int)((bestShift >> 32) & LOWORD);
//			shiftLi = (int)((bestShift      ) & LOWORD);
			final int shiftToId = tgHierarchy[shiftLv].NDgetGraphShiftIndex(shiftLi);
			boolean isSpawn=false;
			
		
			
			// check if it's a spawn shift, otherwise do a traditional s/m shift
			if ((shiftLv >= minSpawnLevel) && (shiftToId == 0))
			{
				isSpawn = true;
				//  this will prevent any spawn shifts from occuring
//				if (true)
//				{
//					shifts.remove(bestShift);
//					continue;
//				}
				
				final int shiftNewClass = spawnShifts.get(bestShift);
				final int shiftFromClass = tgHierarchy[shiftLv].NDgetClassIndex(shiftLi);
				final int shiftFromParentId = tgHierarchy[shiftLv].NDgetParent(shiftLi);
				
				if ((verboseB) && (shiftCount % verboseF == 0) )
					o.printf("[%d] found the best shift (spawn) lv %d id %d with weight of %f class %d->%d\n",shiftCount,
						shiftLv,shiftLi,bestShiftWeight,shiftFromClass,shiftNewClass);
				
				//o.printf("neighbor and length information of the current graph at this node:\n");
				//tgHierarchy[shiftLv].printNeighbors(shiftLi);
				
				//final int tlbl = computeTopLevelBoundaryLength();
				//o.printf("total boundary length at top level (-1) before is %d\n",tlbl);
				//int mac = 2*computeSingleBoundaryLength(shiftLv, shiftLi, shiftFromClass);
				//o.printf("length of the spawning node to neighbors of its old class is %d -- summing %d\n",mac,mac+tlbl);
//				ObjectToDisk.save("/tmp/shift-pre.object", hierarchy);
				
				// change class of shiftLi and all of its children
				pushClassIndexDown(shiftLv,shiftLi,shiftNewClass);
				
//				o.printf("after pushing class index down, class is %d\n",tgHierarchy[shiftLv].NDgetClassIndex(shiftLi));
				
				// effecting the spawn means creating a new tree to the root starting with this level.
				tgHierarchy[shiftLv+1].NDremoveChildId(shiftFromParentId, shiftLi);
				int spawnC = shiftLi;
				int spawnP;
				int mass,intensity;
				mass      = tgHierarchy[shiftLv].NDgetMass(shiftLi);
				intensity = tgHierarchy[shiftLv].NDgetIntensity(shiftLi);
				for (int spawnLevel=shiftLv+1; spawnLevel < tgHierarchy.length; spawnLevel++)
				{
					ThinGraphNode n = tgHierarchy[spawnLevel].addNewNode();
					spawnP = n.getId();
					tgHierarchy[spawnLevel].NDaddChild(spawnP, spawnC);
					tgHierarchy[spawnLevel-1].NDsetParent(spawnC, spawnP);
					tgHierarchy[spawnLevel].NDsetMass(spawnP, mass);
					tgHierarchy[spawnLevel].NDsetIntensity(spawnP, (byte)intensity);
					tgHierarchy[spawnLevel].NDsetClassIndex(spawnP, (byte)shiftNewClass);
					
					if (verboseB)
						o.printf("spawning new node at level %d id %d\n",spawnLevel,spawnP);
					
					// create neighborhood connecting at this new level if it is below the root level (no neighbors at roots)
					if (spawnLevel < tgHierarchy.length - 1)
					{
//						o.printf("creating edge connectivity at level %d\n",spawnLevel);
						
						tempIIHM.clear();
						tempIIHM.put(spawnP,0);
						
						for (int ci=0;ci<n.getNumberOfChildren();ci++)
						{
							int cinn=tgHierarchy[spawnLevel-1].NDgetNumberOfNeighbors(spawnC);
							for (int cin=0;cin<cinn;cin++)
							{
								int cinid = tgHierarchy[spawnLevel-1].NDgetNeighbor(spawnC, cin);
								int len = tgHierarchy[spawnLevel-1].NDgetBoundaryLength(spawnC, cin);
								// now we have to check the parent index	
								int cip = tgHierarchy[spawnLevel-1].NDgetParent(cinid);
							    if (!tempIIHM.adjustValue(cip,len))
							    	tempIIHM.put(cip,len);	
							}
						}
						TIntIntIterator iii = tempIIHM.iterator();
						for (int i = tempIIHM.size(); i-- > 0;) 
						{
							iii.advance();

							int pid = iii.key();
							int len = iii.value();
							if (pid != spawnP)
							{
//								o.printf("level %d creating edge between %d and %d of length %d\n",spawnLevel,spawnP,pid,len);
								tgHierarchy[spawnLevel].addEdge(spawnP,pid,len);
								
								
								// i think the code below should only be called at the level just above the shiftlv
								// or the shiftFromParentId needs to be updated somehow.
								// because the subtract to top is called as it should be at the end of the function...
								// so, I added that in here...
								
								if (spawnLevel == shiftLv+1)
								{
									// have to remove the boundary length from the fromParent to pid
//									System.out.printf("problem place:shiftFromParentId %d, number on spawnLevel %d, spawnLevel %d\n",
//											shiftFromParentId,tgHierarchy[spawnLevel].getNumberOfNodes(),spawnLevel);
									int t = tgHierarchy[spawnLevel].NDlookupNeighborIndex(shiftFromParentId, pid);
									if (t != -1)
									{	
//										o.printf("level %d removing boundary length b/w %d and %d of len %d\n",
//												spawnLevel,shiftFromParentId,pid,len);
										tgHierarchy[spawnLevel].NDsubtractFromBoundaryLength(shiftFromParentId, t, len);
									}
									t = tgHierarchy[spawnLevel].NDlookupNeighborIndex(pid,shiftFromParentId);
									if (t != -1)
									{	
//										o.printf("level %d removing boundary length b/w %d and %d of len %d\n",
//												spawnLevel,pid,shiftFromParentId,len);
										tgHierarchy[spawnLevel].NDsubtractFromBoundaryLength(pid, t, len);
									}	

									updateSubtractToTop(spawnLevel, shiftFromParentId, pid, len);
								}
							}
						}
					}

					spawnC = spawnP;
				}
				spawnP = tgHierarchy[shiftLv].NDgetParent(shiftLi);
				
				// now that the hierarchy is correclty modified, we need to update the likelihood caches
				updateLikelihoodCachesUp(shiftLv,shiftLi,shiftFromParentId,spawnP);
				
				computeShiftCounter=0;
				shiftsTouched.clear();
				
//				o.printf("total boundary length at top level (-1) after is %d\n",computeTopLevelBoundaryLength());
//				ObjectToDisk.save("/tmp/shift-post.object", hierarchy);
				
				// now we update the shifts up the hierarchy
				if (shiftLv < startLevel)
				{		
					updateShiftsUp(shiftLv+1,spawnP,startLevel);
					updateShiftsUp(shiftLv+1,shiftFromParentId,startLevel);
				}  
			}
			else
			{

				int shiftFromClass, shiftToClass, shiftToParentId, shiftFromParentId;
				shiftFromParentId = tgHierarchy[shiftLv].NDgetParent(shiftLi);
				shiftToParentId = tgHierarchy[shiftLv].NDgetParent(shiftToId);
				shiftFromClass = tgHierarchy[shiftLv].NDgetClassIndex(shiftLi);
				shiftToClass = tgHierarchy[shiftLv].NDgetClassIndex(shiftToId);

				if (shiftFromClass == shiftToClass)
				{
//					o.printf("[%d] skipping shift (outdated due to previous shifts, but not removed from list)\n",--shiftCount);
					shifts.remove(bestShift);
					continue;
				}

				if ((verboseB) && (shiftCount % verboseF == 0) )
					o.printf("[%d] found the best shift lv %d id %d->%d parent %d->%d with weight of %f class %d->%d\n",shiftCount,
							shiftLv,shiftLi,shiftToId,shiftFromParentId,shiftToParentId,bestShiftWeight,shiftFromClass,shiftToClass);


				// change class of shiftLi and all of its children
				pushClassIndexDown(shiftLv,shiftLi,shiftToClass);

				// change the parent of the node
				tgHierarchy[shiftLv].NDsetParent(shiftLi, tgHierarchy[shiftLv].NDgetParent(shiftToId));
				tgHierarchy[shiftLv+1].NDaddChild(shiftToParentId, shiftLi);
				tgHierarchy[shiftLv+1].NDremoveChildId(shiftFromParentId, shiftLi);

//				if (0 >= tgHierarchy[shiftLv+1].NDgetNumberOfChildren(shiftFromParentId))
//				{
////				o.printf("the from parent has no more children;  should be deleted.\n");
//				}

				///  --- 
				computeShiftCounter=0;
				shiftsTouched.clear();

				// the first block updates information in the hierarchy at and above the shift level
				if (shiftLv < tgHierarchy.length-2)
				{
//					o.printf("updating boundary lengths up\n");
					updateBoundaryLengthsUpSmart(shiftLv,shiftLi,shiftFromParentId,shiftToParentId);
//					o.printf("updating likelihood caches up\n");
					updateLikelihoodCachesUp(shiftLv,shiftLi,shiftFromParentId,shiftToParentId);
				}

				// this second block updates the shifts at and above the shift level
				if (shiftLv < startLevel)
				{
//					o.printf("updating shifts up FromParent\n");
					updateShiftsUp(shiftLv+1,shiftFromParentId,startLevel);
//					o.printf("updating shifts up ToParent\n");
					updateShiftsUp(shiftLv+1,shiftToParentId,startLevel);
				}

			} // end of s/m shift
			
//			o.printf("updating shifts down\n");

			updateShiftsDownCount=0;
			updateShiftsDown(shiftLv,shiftLi);
			
			// this is a safety call;  there is no time that the a shift for a node should remain active
			//  after being taken --> when it's a -1 in the shift index store
			int safety1 = tgHierarchy[shiftLv].NDgetGraphShiftIndex(shiftLi);
			if (safety1 < 0)
				shifts.remove(bestShift);

			if ((verboseB) && (shiftCount % verboseF == 0) )
				o.printf("%s Class:  updated shifts set: %d remain (%.3f%%)\t%d attempts\n",this.getClass().getSimpleName(),
					shifts.size(),100.0f*(float)shifts.size()/(float)numPotentialShifts,updateShiftsDownCount);
			
//			checkBoundaryLengths();  //debug
			
			if (haveObservers)
				fireShiftToObservers(shiftCount, shiftLv, shiftLi, shiftToId, bestShiftWeight);
		}
		
		fireShiftEndToObservers();
		
		if (pw != null)
			pw.close();
		
	}
	
	public void fireShiftBeginToObservers()
	{
		for (GraphShiftsObserver O : observers)
			O.shiftBegin();
	}
	
	public void fireShiftEndToObservers()
	{
		for (GraphShiftsObserver O : observers)
			O.shiftEnd();
	}
	
	public void fireShiftToObservers(int shiftNumber, int shiftLevel, int shiftFrom, int shiftTo,  float shiftWeight)
	{
		for (GraphShiftsObserver O : observers)
			O.shiftTaken(shiftNumber, shiftLevel, shiftFrom, shiftTo, shiftWeight, hierarchy, this);
	}
	
	public byte[] getClassIndexArray()
	{
		return tgHierarchy[0].getClassIndexBuffer().toNativeArray();
	}
	
	
	public ScalarImageB3 getFinalLabels()
	{
		ScalarImageB3 I = new ScalarImageB3(w,h,d);
		
		byte Fdat[] = tgHierarchy[0].getClassIndexBuffer().toNativeArray();
		byte Ibuf[] = I.getBuffer().getData();
		System.arraycopy(Fdat,0,Ibuf,0,Ibuf.length);
		
		return I;
	}
	
	public void initializeGraph(GraphShiftsInitializer I, boolean makeSpawn, double pruneSpawn)
	{
		o.printf("%s Class:  initializing the hierarchy\n",this.getClass().getSimpleName());
		hierarchy = I.initializeGraph(image);
		o.printf("%s Class:  initialized with %d levels\n",this.getClass().getSimpleName(),hierarchy.getNumberOfLevels());
		
		o.printf("%s Class:  creating levels array of ThinGraph\n", this.getClass().getSimpleName());
		tgHierarchy = new ThinGraph[hierarchy.getNumberOfLayers()];
		for (int i=0;i<tgHierarchy.length;i++)
			tgHierarchy[i] = (ThinGraph)hierarchy.getLayer(i);
		
//		o.printf("%s Class:  writing the initial hierarchy to disk\n",this.getClass().getSimpleName());
//		ObjectToDisk.save("/tmp/hierarchy.object", hierarchy);
//		o.printf("%s Class:  done initialization\n",this.getClass().getSimpleName());
//		
//		o.printf("%s Class:  writing the initial labels to disk\n",this.getClass().getSimpleName());
//		ScalarImageB3 labels = getFinalLabels();
//		AnalyzeHeader hdr = new AnalyzeHeader("/tmp/initial-labels",labels);
//		hdr.saveToDisk();
		o.printf("%s Class:  done initialization\n",this.getClass().getSimpleName());
		

		/*** -------------------  Initialize the likelihood values --------------- ***/	
		for (int i=1;i<tgHierarchy.length;i++)
		{
			ThinGraph gi  = tgHierarchy[i];
			o.printf("initializing likelihood cache at level %d\n", i);
			
			int N = gi.getNumberOfNodes();
			for (int j=(i >= minSpawnLevel)?1:0;j<N;j++)
			{
				for (int c=0;c<numClasses;c++)
				{
					double L = computeLLAtLeavesCache(i, j, c);
					gi.NDsetLikelihood(j, c, (float)L);
				}
			}
		}
		
		
		/*** -------------------- Make/Prune Spawn Edges ------------------------------- ***/
		/*** original code used to make the spawn edges in the bottom-up code and then prune them here
		 *    but, it's more efficient to delay construction until here.  THis will also check if the user
		 *    wants to make these edges at all.  (not making these edges is how we do only the split/merge
		 *    shifts....
		 */
		if (makeSpawn)
		{
			double ltau = Math.log(pruneSpawn);
			
			for (int i=minSpawnLevel;i<tgHierarchy.length-1;i++)
			{
				int numrm=0;
				int numad=0;

				ThinGraph gi = tgHierarchy[i];
				int n = gi.getNumberOfNodes();
				for (int ni=1;ni<n;ni++)
				{
					int clid = gi.NDgetClassIndex(ni);
					double L = gi.NDgetLikelihood(ni, clid);
					if (L / gi.NDgetMass(ni) > ltau)
					{
						//gi.removeEdge(0, ni);
						numrm++;
					}
					else
					{
						gi.addEdge(0, ni,0);
						numad++;
					}
				}
				System.out.printf("Level %d, %d of %d Spawn Connections Added (%d not)\n",i,numad,n,numrm);
			}
		}
		
	}
	
	
	
	final private long packShift(int level, int nodeId)
	{
		return ((((long)level) << 32) & HIWORD) | (((long)nodeId) & LOWORD);
	}
	
	
	private void pushClassIndexDown(int level, int nodeId, int classIdx)
	{
		tgHierarchy[level].NDsetClassIndex(nodeId,classIdx);
		if (level == 0)
			return;
		
		int clen = tgHierarchy[level].NDgetNumberOfChildren(nodeId);
		for (int i=0;i<clen;i++)
		{
			int ci = tgHierarchy[level].NDgetChild(nodeId, i);
			pushClassIndexDown(level-1,ci,classIdx);
		}
	}
	
	
	
	public void removeObserver(GraphShiftsObserver O)
	{
		observers.remove(O);
		if (observers.size()==0)
			haveObservers=false;
	}
	
	/**
	 * This unpacks the shift argument into a level and a nodeId that are stored
	 *  as members of the enclosing class to avoid needing to make some class and return
	 *  the from this function.
	 *  level is named unpackedLevel
	 *  nodeId is named unpackedNodeId
	 * It should be assumed that these values are only valid immediately after this function call.
	 * NOT SYNCHRONIZED
	 * @param shift
	 */
	final private void unpackShift(long shift)
	{
		unpackedLevel  = (int)((shift >> 32) & LOWORD);
		unpackedNodeId = (int)((shift      ) & LOWORD);
	}
	final private void updateAddToTop(int level,int uid,int vid,int len)
	{
		if (level < tgHierarchy.length-2)
		{
			int upid = tgHierarchy[level].NDgetParent(uid);
			int vpid = tgHierarchy[level].NDgetParent(vid);
			if (upid != vpid)
			{
				int u = tgHierarchy[level+1].NDlookupNeighborIndex(upid, vpid);
				if (u == -1)
				{
					tgHierarchy[level+1].addEdge(upid, vpid, len); // will add this edge both ways
				}
				else
				{
					int v = tgHierarchy[level+1].NDlookupNeighborIndex(vpid, upid);
					tgHierarchy[level+1].NDaddToBoundaryLength(upid,u,len);
					tgHierarchy[level+1].NDaddToBoundaryLength(vpid,v,len);
				}
				updateAddToTop(level+1,upid,vpid,len);
			}
		}
	}
	
	/**
	 * A faster version of the update boundary lengths up...less calls, only go through children once
	 * @param level
	 * @param uid
	 * @param vid
	 */
	private void updateBoundaryLengthsUp(int level, int uid)
	{
		
		ThinGraph gi  = tgHierarchy[level];
		ThinGraph gi_ = tgHierarchy[level-1];

		int cn = gi.NDgetNumberOfChildren(uid);
		boundaryLengthIIHM.clear();

		for (int cid=0;cid<cn;cid++)
		{
			int ci = gi.NDgetChild(uid, cid);
			int cin = gi_.NDgetNumberOfNeighbors(ci);

			for (int cini=0;cini<cin;cini++)
			{
				int nid = gi_.NDgetNeighbor(ci, cini);
				int pid = gi_.NDgetParent(nid);

				// if we have the same parent then buzz-off
				if (pid == uid)
					continue;

				int len = gi_.NDgetBoundaryLength(ci, cini);

				if (!boundaryLengthIIHM.adjustValue(pid,len))
					boundaryLengthIIHM.put(pid,len);
			}
		}   // end of looping through uid's children

		int nn = gi.NDgetNumberOfNeighbors(uid);
		for (int ni=0;ni<nn;ni++)
		{
			int nid = gi.NDgetNeighbor(uid, ni);
			int len = boundaryLengthIIHM.get( nid );
			gi.NDsetBoundaryLength(uid, ni, len);
			
			int u = gi.NDlookupNeighborIndex(nid, uid);
			gi.NDsetBoundaryLength(nid, u, len);
		}
		
		if (level < tgHierarchy.length-2)
		{
			updateBoundaryLengthsUp(level+1,tgHierarchy[level].NDgetParent(uid));
		}
	}
	
	/**  update boundary lengths at level2 by using class labels of children and their boundary lengths.
	 * ...this is mutually from uid to all of its neighbors.  the total change in the length
	 * should equal deltaLength.
	 * If we do not hit vid as a neighbor, then we have to create an edge from uid to vid and compute 
	 *  the length on it too.
	 * 
	 * The function computeBoundaryLengthToNeighborViaChildren is to be used
	 * All length updates are symmetric.
	 * 
	 * TODO   This could be written smarter to compute len to all neighbors at once like it is done initially
	 *  in the EdgeSamplerInitializer.
	 */
	private void updateBoundaryLengthsUpOLD(int level, int uid, int vid)
	{
		boolean hitVid=false;
		int total=0;
		
		int n = tgHierarchy[level].NDgetNumberOfNeighbors(uid);
		for (int i=0;i<n;i++)
		{
			int nid = tgHierarchy[level].NDgetNeighbor(uid, i);
			// I initially thought that I only had to compute it against neighbors of the to Class, but that
			//  appears to be wrong and I need to inspect deeper to find out why.
//			if (tgHierarchy[level].NDgetClassIndex(nid) != toCid)
//				continue;
//			
			if (nid == vid)
				hitVid = true;
			
			int oldlen = tgHierarchy[level].NDgetBoundaryLength(uid, i);
			
			int len = computeBoundaryLengthToNeighborViaChildren(level, uid, nid);
			tgHierarchy[level].NDsetBoundaryLength(uid, i, len);
			int u = tgHierarchy[level].NDlookupNeighborIndex(nid, uid);
			tgHierarchy[level].NDsetBoundaryLength(nid, u, len);
//			System.out.printf("\t[%d]during bl up, set len from %d to %d (vv) to len %d (oldlen is %d)\n",level,uid,nid,len,oldlen);
			total += len;
		}
		
		if (!hitVid)
		{
			int len = computeBoundaryLengthToNeighborViaChildren(level, uid, vid);
			total += len;
			tgHierarchy[level].addEdge(uid, vid, len);
//			System.out.printf("during bl up, had to add edge %d to %d len %d\n",uid,vid,len);
		}
		
//		System.out.printf("\t[%d] after bl up, set total length to %d\n",level,total);
		if (level < tgHierarchy.length-2)
		{
			updateBoundaryLengthsUpOLD(level+1,tgHierarchy[level].NDgetParent(uid),
					                         tgHierarchy[level].NDgetParent(vid));
		}
	}
	/**
	 * Avoid recomputing entire boundary lengths for parents...
	 * Just carefully compute the changes in lengths due to the shift
	 * @param level
	 * @param uid
	 * @param fpid
	 * @param tpid
	 */
	public void updateBoundaryLengthsUpSmart(int level,int uid,int fpid,int tpid)
	{
//		o.printf("\tubl new for node %d on level %d fpid %d tpid %d\n",uid,level,fpid,tpid);
		ThinGraph g = tgHierarchy[level];
		ThinGraph gg = tgHierarchy[level+1];
		
		int lenToRemoveFromTPID=0;
		boundaryLengthIIHM.clear();
		
		int n = g.NDgetNumberOfNeighbors(uid);
		for (int ni=0;ni<n;ni++)
		{
			int nid = g.NDgetNeighbor(uid, ni);
			int npid = g.NDgetParent(nid);
			if (npid == tpid)
			{
//				o.printf("\tuid ngb %d, causing removal of %d from length\n",nid,g.NDgetBoundaryLength(uid, ni));
				lenToRemoveFromTPID += g.NDgetBoundaryLength(uid, ni);
			}
			else
			{
//				o.printf("\tuid ngb %d, causing adding of %d from length\n",nid,g.NDgetBoundaryLength(uid, ni));
				int len = g.NDgetBoundaryLength(uid,ni);
				if (!boundaryLengthIIHM.adjustValue(npid,len))
					boundaryLengthIIHM.put(npid,len);	
			}
		}
		
	    int tnid = gg.NDlookupNeighborIndex(fpid, tpid);
	    int fnid = gg.NDlookupNeighborIndex(tpid, fpid);
		gg.NDsubtractFromBoundaryLength(tpid,fnid,lenToRemoveFromTPID);
		gg.NDsubtractFromBoundaryLength(fpid,tnid,lenToRemoveFromTPID);
		updateSubtractToTop(level+1,tpid,fpid,lenToRemoveFromTPID);
		
//		o.printf("ublNEW:  subtracting between %d and %d  %d\n", tpid,fpid,lenToRemoveFromTPID);
		
		TIntIntIterator iii = boundaryLengthIIHM.iterator();
		for (int i = boundaryLengthIIHM.size(); i-- > 0;) 
		{
		   iii.advance();
		   
		   int pid = iii.key();
		   int len = iii.value();
		   
		   int pnid = gg.NDlookupNeighborIndex(tpid, pid);
		  
		   if (pnid == -1)
		   {
			   // have to add the edge
			   gg.addEdge(tpid, pid, len); // will add this edge both ways
//			   o.printf("ublNEW:  adding new edge between %d and %d  %d\n", tpid,pid,len);
		   }
		   else
		   {
			   gg.NDaddToBoundaryLength(tpid, pnid, len);
			   int t = gg.NDlookupNeighborIndex(pid, tpid);
			   gg.NDaddToBoundaryLength(pid, t, len);
			   
//			   o.printf("ublNEW:  adding between %d and %d  %d\n", tpid,pid,len);
		   }
		   
		   updateAddToTop(level+1, tpid, pid, len);
		   
		   // we have to also remove this len from the neighbor of fpid to keep things kosher
		   fpidpidupdate:
		   if (fpid != pid) // can happen with highly connected situations
		   {   
//			   o.printf("ublNEW:  subtracting between %d and %d  %d\n", fpid,pid,len);
			   int f = gg.NDlookupNeighborIndex(fpid, pid);
			   if (f == -1)
			   {
				   break fpidpidupdate;
			   }
			   gg.NDsubtractFromBoundaryLength(fpid, f, len);
			   f = gg.NDlookupNeighborIndex(pid, fpid);
			   if (f == -1)
			   {
				   break fpidpidupdate;
			   }
			   gg.NDsubtractFromBoundaryLength(pid, f, len);

			   updateSubtractToTop(level+1, fpid, pid, len);
		   }
		}
		
	}
	
	/** 
	 * After a shift occurs, we may need to add neighbors in the to parent (uid) because the new
	 *  child may induce neighbors for the parent that did not exist before. 
	 * @param level
	 * @param uid
	 * @param cgid   child graph node id on level -1
	 */
	private void updateFindNewNeighborsViaChild(int level, int uid, int cgid)
	{
		int total=0;
		
		
		newNgbFromChildIHS.clear();
		
		int ncn = tgHierarchy[level-1].NDgetNumberOfNeighbors(cgid);
		for (int ncni=0;ncni<ncn;ncni++)
		{
		  int ncnid = tgHierarchy[level-1].NDgetNeighbor(cgid, ncni);
		  int ncnip = tgHierarchy[level-1].NDgetParent(ncnid);
		  if (ncnip == uid)
			  continue;
		  
		  if (!newNgbFromChildIHS.contains(ncnip))
		  {
			  int nid = tgHierarchy[level].NDlookupNeighborIndex(uid, ncnip);
			  if (nid==-1)
			  {
//				  int len = computeBoundaryLengthToNeighborViaChildren(level, uid, ncnip);
				  int len = 1;
				  total += len;
				  tgHierarchy[level].addEdge(uid, ncnip, len);
//				  System.out.printf("during blnew children up, had to add edge %d to %d len %d\n",uid,ncnip,len);
			  }
			  newNgbFromChildIHS.add(ncnip);
		  }
		}
		
//		if (level < tgHierarchy.length-2)
//		{
//			TIntIterator it = newNgbFromChildIHS.iterator();
//			int uidp = tgHierarchy[level].NDgetParent(uid);
//			while (it.hasNext())
//			{
//				int p = it.next();
//				updateFindNewNeighborsViaChild(level+1,uidp,uid);
//			}
//		}
	}
	
	/* the likelihood values for shiftLi must be transfered from the FromParent to the ToParent
	 */
	protected void updateLikelihoodCachesUp(int shiftLv,int shiftLi,int shiftFromParentId,int shiftToParentId)
	{
		if (shiftLv == 0)
		{
			for (int c=0;c<numClasses;c++)
			{
				LBUF[c] = (float)computeLLAtLeavesCache(0,shiftLi,c);
				tgHierarchy[shiftLv+1].NDsubtractFromLikelihood(shiftFromParentId, c, LBUF[c]);
				tgHierarchy[shiftLv+1].NDaddToLikelihood(shiftToParentId, c, LBUF[c]);
			}
		}
		else
		{
			for (int c=0;c<numClasses;c++)
			{
				LBUF[c] = tgHierarchy[shiftLv].NDgetLikelihood(shiftLi, c);
				tgHierarchy[shiftLv+1].NDsubtractFromLikelihood(shiftFromParentId, c, LBUF[c]);
				tgHierarchy[shiftLv+1].NDaddToLikelihood(shiftToParentId, c, LBUF[c]);
			}
		}
		int lvl = shiftLv + 1;
		int fp = shiftFromParentId;
		int tp = shiftToParentId;
		while (lvl<tgHierarchy.length-2)
		{
			fp = tgHierarchy[lvl].NDgetParent(fp);
			tp = tgHierarchy[lvl].NDgetParent(tp);
			if (fp == tp)
				break;
			lvl++;
			for (int c=0;c<numClasses;c++)
			{
				tgHierarchy[lvl].NDsubtractFromLikelihood(fp, c, LBUF[c]);
				tgHierarchy[lvl].NDaddToLikelihood(tp, c, LBUF[c]);
			}
		}
	}
	
	/**
	 * This is a recursive function that will update the shift for the given node and its neighbors.
	 * Then, it will issue this same function call to all of its children.
	 * 
	 * Updating a shift means 
	   1 Remove any shift s currently associated with n from the shift hashset. (if necessary0
	   2 Remove the shift information from n.
	   3 Recompute the shift on n given the current state of the graph.
	   4   For each neighbor m of n compute the shift s and compute the best s.
	   5   If there is no best shift (all are invalid), then no shift will be added for s.
	   6 For each neighbor m of n, update the shift
       7   If the shift on m was to n, then we need to completely revaluate the shift for m.
	   8   Else if the shift on m was to some n' != n, then, simply evaluate potential shift to n and store whichever is best.
	   9 Recurse to children.	
	 * @param level  The int level in the graph hierarchy the node is on.
	 * @param nodeId  The id of the node in the graph hierarchy.
	 */
	private void updateShiftsDown(int level, int nodeId)
	{
		long shift = packShift(level,nodeId);
		shiftsTouched.add(shift);
		
		updateShiftsDownCount++;
		int shiftTo = computeShift(level,nodeId);
		
		// if the shiftTo is -1 or -2, then there is no shift for this guy remaining
		//  so, we should remove any shift it has from the hashset (it's shift of -1,0 would have been set
		//  in the computeShifts function)
		if (shiftTo >= 0)
			shifts.add(shift);  // either it was there already or we're putting it there now
		else
		{
			// check if this bad shift came because we and all of our neighbors share the same class
			//  if so, then, we need to check if there was a shift for this node...if so, then we
			//   go through to its neighbors and children to clear them from the shifts list
			//   otherwise, we just return
//			if (!shifts.contains(shift) && (shiftTo == -2))
//				return;
//			
			// ottherwise, just remove it and proceed to the neighdors and children
			shifts.remove(shift);
		}	
		
		int uCI = tgHierarchy[level].NDgetClassIndex(nodeId);
		int unN = tgHierarchy[level].NDgetNumberOfNeighbors(nodeId);
		for (int uni=0;uni<unN;uni++)
		{
			int neighborId = tgHierarchy[level].NDgetNeighbor(nodeId, uni);
			
			// skip over this guy if it's a spawn node
			if ((level >= minSpawnLevel) && (neighborId == 0))
				continue;
			
			// check the class id here first and skip if they're the same
			if (uCI == tgHierarchy[level].NDgetClassIndex(neighborId))
			{
				shifts.remove(packShift(level,neighborId)); // need to remove it just in case it was there
				continue;
			}
			
			shift = packShift(level,neighborId);
			if (!shiftsTouched.contains(shift))  // this get ride of over half the calls... good
			{
				shiftTo = computeShift(level,neighborId);

				shift = packShift(level,neighborId);
				if (shiftTo >= 0)
					shifts.add(shift);  // either it was there already or we're putting it there now
				else 
					shifts.remove(shift);
			}
		}
		
		int clen = tgHierarchy[level].NDgetNumberOfChildren(nodeId);
		for (int i=0;i<clen;i++)
		{
			int ci = tgHierarchy[level].NDgetChild(nodeId, i);
			updateShiftsDown(level-1,ci);
		}
	}
	
	/** When a shift is made at level k, then the old and new parents must also update their
	 *   shift values -- since their respective subtrees have changed.
	 *  The updateBoundaryLengthsUp is assumed to be called first...
	 * @param level
	 * @param nodeId
	 */
	private void updateShiftsUp(int level, int nodeId,int stopLevel)
	{
		long shift = packShift(level,nodeId);
		shiftsTouched.add(shift);
		
		int shiftTo = computeShift(level,nodeId);
		
		// if the shiftTo is -1, then there is no shift for this guy remaining
		//  so, we should remove any shift it has from the hashset (it's shift of -1,0 would have been set
		//  in the computeShifts function)
		if (shiftTo == -1)
			shifts.remove(shift);
		else 
			shifts.add(shift);  // either it was there already or we're putting it there now
		
		
		// we don't need to update the shifts on the neighbors because no ones parent has changed...
		
		
		if (level < stopLevel)
		{
			int p = tgHierarchy[level].NDgetParent(nodeId);
			if (!shiftsTouched.contains(packShift(level+1,p)))
				updateShiftsUp(level+1,p,stopLevel);
		}
	}
	
	/**
	 * Update shifts up  is a recursive function that updates the shift values (modifies them, not
	 *  replaces them) from the current shift level and node to the toplevel of the graph.  This only
	 *  updates those values that are "global" and do not compute their shift values from the leaves
	 *  directly, but instead store them at the actual node.  
	 * The energies that are updated are:
	 *  Boundary Smooth term.  -- we update both the change in the length of the boundary as
	 *                             well as the change in Esmooth if this is the active shift 
	 */
	private void updateShiftsUpOLD(int level, int uid, int vid, int deltaLength)
	{
		float deltaWeight = (float)(deltaLength * lambdaSmooth);
		
		// find the neighbor index for u to v 
		updateUandV:
		{
			int vin = tgHierarchy[level].NDlookupNeighborIndex(uid, vid);
			if (vin != -1)
			{
				// update the boundary length
				int len = tgHierarchy[level].NDgetBoundaryLength(uid, vin);
				tgHierarchy[level].NDsetBoundaryLength(uid, vin, len + deltaLength); 

				// check if the shift is this guy...
				int sid = tgHierarchy[level].NDgetGraphShiftIndex(uid);
				if (sid == vid)
				{
					float w = tgHierarchy[level].NDgetGraphShiftWeight(uid);
					tgHierarchy[level].NDsetGraphShiftWeight(uid, w+deltaWeight);
				}
			}
			// when this case occurs, it is because we need to update the connectivity
			//  of the graph.  "Enough" nodes have been reassigned that we need to make new
			//  neighbors between two nodes that were previously not neighbors.
			else
			{
				// we need to make a new edge in the graph connecting uid and vid
				tgHierarchy[level].addEdge(uid, vid, deltaLength);

				// if we get here, then we don't need to do do the v->u update
				break updateUandV;
			}

			// find the neighbor index for v to u
			int uin = tgHierarchy[level].NDlookupNeighborIndex(vid, uid);
			if (uin != -1)
			{
				// update the boundary length
				int len = tgHierarchy[level].NDgetBoundaryLength(vid, uin);
				tgHierarchy[level].NDsetBoundaryLength(vid, uin, len + deltaLength); 

				// check if the shift is this guy...
				int sid = tgHierarchy[level].NDgetGraphShiftIndex(vid);
				if (sid == uid)
				{
					float w = tgHierarchy[level].NDgetGraphShiftWeight(vid);
					tgHierarchy[level].NDsetGraphShiftWeight(vid, w+deltaWeight);
				}
			}
			// the else case should never happen, this would be a bug
			else
			{
				o.printf("ERROR #2\n");
				// throw new AssertionError("Error on updateShiftsUp_aux, lookup failed for neighbor, shouldn't happen\n");
			}

		}
		
		
		if (level < tgHierarchy.length-2)
		{
			updateShiftsUpOLD(level+1,tgHierarchy[level].NDgetParent(uid),
					               tgHierarchy[level].NDgetParent(vid),deltaLength);
		}
	}
	
	final private void updateSubtractToTop(int level,int uid,int vid,int len)
	{
		if (level < tgHierarchy.length-2)
		{
			int upid = tgHierarchy[level].NDgetParent(uid);
			int vpid = tgHierarchy[level].NDgetParent(vid);
			if (upid != vpid)
			{
				int u = tgHierarchy[level+1].NDlookupNeighborIndex(upid, vpid);
				int v = tgHierarchy[level+1].NDlookupNeighborIndex(vpid, upid);
				if (u != -1)
					tgHierarchy[level+1].NDsubtractFromBoundaryLength(upid,u,len);
				if (v != -1)
					tgHierarchy[level+1].NDsubtractFromBoundaryLength(vpid,v,len);
				updateSubtractToTop(level+1,upid,vpid,len);
			}
		}
	}

}

