package edu.ucla.ccb.graphshifts.graphs;

import edu.ucla.ccb.graphshifts.image.Image3PixelAccess;
import gnu.trove.TByteArrayList;
import gnu.trove.TFloatArrayList;
import gnu.trove.TIntArrayList;
import gnu.trove.TLongArrayList;

import java.io.PrintStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;


/**
 * The ThinGraph is an attempt to store a complete instance of a GraphModel structure 
 *  without making it necessary to take up a HUGE memory footprint like my other graph 
 *  classes (e.g. UndirectedGraph* and the GraphPyramid.GraphLayer).  The major 
 *  observation here is that the Java construct for creating objects requires a per-object 
 *  overhead that makes creating graphs for large datasets too expensive.  Secondary 
 *  observations include the need to reduce the amount of references to objects that we 
 *  have floating around so that any free objects can be garbage collected and serializing
 *  objects is trivial (and not affected by the recursion bug).  
 *   
 * In this class, we make some assumptions
 *  1. The nodes are sequentially indexed and their id's are implicit. 
 *  2. All link relations (neighbor, parent, children) are stored as integer indices (see 1.).
 *  
 * Remarks:
 *  1. To make it plausible to store only primitive types to encode the link information at 
 *      each node, I had to find/create an IntArrayList class.  I use the GNU Trove library
 *      for this purpose.  
 *  2. While I make it plausible to dynamically add nodes (no removing), doing so will cause much
 *      memory rearrangement.  It is thus better to give the number of nodes at the outset...
 *  
 */
public class ThinGraph implements MutableGraphModel, Serializable
{
	
	public static class ThinGraphNode implements GraphNodeModel, SingleParentProtocol, GraphNodeDebuggable
	{
		private ThinGraph owner;
		private int id;
		public ThinGraphNode(ThinGraph owner, int id)
		{
			this.owner = owner;
			this.id = id;
		}
		public void addChild(ThinGraphNode n)
		{
			owner.NDaddChild(id,n.id);
		}
		
		public void debugOut(PrintStream s)
		{
			s.printf("graph node %d debug information\n", id);
			
			int nn = getNumberOfNeighbors();
			s.printf("neighbor information (%d neighbors)\n", nn);
			for (int i=0;i<nn;i++)
			{
				s.printf("\t[n %d]  to %d  len %d\n",i,getNeighborId(i),getBoundaryLength(i));
			}
			
		}

		public int getBoundaryLength(int i)
		{
			return owner.NDgetBoundaryLength(id,i);
		}

		/** We can only return an integer from this class at this time.  If it is
		 * determined that we need to actually return the GraphNodeModel object to 
		 * conform to a future protocol, then we will need to store the child-graph
		 * as well.
		 * @param i
		 * @return
		 */
		public int getChild(int i)
		{
			return owner.NDgetChild(id,i);
		}
		
		public int getClassIndex()
		{
			return owner.NDgetClassIndex(id);  
		}
		public Object getData()
		{
			return null;
		}
		public int getGraphShiftIndex()
		{
			return owner.NDgetGraphShiftIndex(id);  
		}
		public float getGraphShiftWeight()
		{
			return owner.NDgetGraphShiftWeight(id);  
		}
		public int getId()
		{
			return id;
		}

		public byte getIntensity()
		{
			return owner.NDgetIntensity(id);
		}

		/* this is a temporary function for dealing with an old scs file during development
		 *  the intensities that I always worked with in the past were float in the range of 0:1
		 *  but storing those floats when they can only take a fixed number of values is really 
		 *  a waste of memory
		 */
		public float getIntensityScaled()
		{
			return ((float)(0xFF&owner.NDgetIntensity(id))) / 255.0f;
		}
		
		public int getIntensityUnsigned()
		{
			return 0xFF & owner.NDgetIntensity(id);
		}
		
		public int getMass()
		{
			return owner.NDgetMass(id);
		}


		/** there may be a bit of confusion over naming here.  In the ThuinGraph class, these 
		 * getNeighbor functions return the id normally and the node when the *Node function is 
		 * called.  But, I had to leave this here like this because of the GraphNodeModel
		 */
		public GraphNodeModel getNeighbor(int i)
		{
			return owner.NDgetNeighborNode(id,i);
		}
		
		public int getNeighborId(int i)
		{
			return owner.NDgetNeighbor(id,i);
		}


		public int  getNumberOfChildren()
		{
			return owner.NDgetNumberOfChildren(id);
		}
		
		public int getNumberOfNeighbors()
		{
			return owner.NDgetNumberOfNeighbors(id);
		}
		
		public int getParent()
		{
			return owner.NDgetParent(id);
		}
		
		/** To suppor the SingleParentProtocol */
		public int getParentIndex()
		{
			return owner.NDgetParent(id);
		}
		
		public void setClassIndex(int classIndex)
		{
			owner.NDsetClassIndex(id,classIndex);
		}

		public void setGraphShift(int sid, float weight)
		{
			owner.NDsetGraphShift(id,sid,weight);  
		}
		
		public void setGraphShiftIndex(int sid)
		{
			owner.NDsetGraphShiftIndex(id,sid);  
		}
		
		
		public void setGraphShiftWeight(float weight)
		{
			owner.NDsetGraphShiftWeight(id,weight);  
		}
		
		private void setId(int id)
		{
			this.id = id;
		}
		
		public void setIntensity(byte intensity)
		{
			owner.NDsetIntensity(id,intensity);
		}

		public void setMass(int mass)
		{
			owner.NDsetMass(id, mass);
		}

		public void setParent(GraphNodeModel node)
		{
			owner.NDsetParent(id, node.getId());
		}
		public void setParent(int parent)
		{
			owner.NDsetParent(id, parent);
		}
		
	}
	static class ThinGraphNodeIterator implements Iterator<GraphNodeModel>
	{
		ThinGraph owner;
		int idx=0;
		
		public ThinGraphNodeIterator(ThinGraph owner)
		{
			this.owner = owner;
		}

		public boolean hasNext()
		{
			return idx < owner.getNumberOfNodes();
		}

		public GraphNodeModel next()
		{
			return owner.getNode(idx++);
		}
		
		/** Nodes cannot be removed
		 *
		 */
		public void remove()
		{
		}
	}
	/** a static buffer variable (what's the right technical term for this?) to store the contents
	 *  of the edge length after unpacking the long edge.
	 */
	static int unpackedLength;
	
	/** a static buffer variable (what's the right technical term for this?) to store the contents
	 *  of the neighbor index after unpacking the long edge.
	 */
	static int unpackedNeighbor;
	
	private static final long HIWORD = 0xFFFFFFFF00000000L;
	private static final long LOWORD = 0x00000000FFFFFFFFL;
	static public ThinGraph createFromVolume(Image3PixelAccess I)
	{
		return createFromVolume(I,9);
	}
	static public ThinGraph createFromVolume(Image3PixelAccess I, int classNum)
	{
		int wh,w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		wh= w*h;
		
		Runtime rt = java.lang.Runtime.getRuntime();
		
		System.out.printf("\t totalmem %d  freemem %d  allocd %012d\n",
				rt.totalMemory(),rt.freeMemory(),rt.totalMemory() - rt.freeMemory());
		
		ThinGraph G = new ThinGraph();
		
		G.allocateMembers(w*h*d, 6, 0);
		G.numberOfNodes = w*h*d;
		G.classNum=classNum;
	    G.lvfiller = new float[classNum];
		
		System.out.printf("\t totalmem %d  freemem %d  allocd %012d\n",
				rt.totalMemory(),rt.freeMemory(),rt.totalMemory() - rt.freeMemory());
		
		G.populateWithDefault(w*h*d);
		
		int i=0;
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
					G.intensity.set(i++,I.getPixelByte(x, y, z));
		
		i=0;
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++,i++)
				{
					int xx,yy,zz,ii;
					TLongArrayList sn = G.neighbors.get(i);
					
					if (x < w-1)
					{
						xx = x+1;
						yy = y;
						zz = z;
						ii = zz*wh+yy*w+xx;
						
						TLongArrayList tn = G.neighbors.get(ii);
						sn.add(packNeighbor(1,ii));
						tn.add(packNeighbor(1,i));
					}
					if (y < h-1)
					{
						xx = x;
						yy = y+1;
						zz = z;
						ii = zz*wh+yy*w+xx;
						
						TLongArrayList tn = G.neighbors.get(ii);
						sn.add(packNeighbor(1,ii));
						tn.add(packNeighbor(1,i));
					}
					if (z < d-1)
					{
						xx = x;
						yy = y;
						zz = z+1;
						ii = zz*wh+yy*w+xx;
						
						TLongArrayList tn = G.neighbors.get(ii);
						sn.add(packNeighbor(1,ii));
						tn.add(packNeighbor(1,i));
					}
				}
		
		
		return G;
	}
	
	/** Defines the convention on how to pack an int edge length and int neighbor index
	 *  into a long variable.
	 * @param length
	 * @param neighbor
	 * @return
	 */
	final static private long packNeighbor(int length,int neighbor)
	{
		return ((((long)length) << 32) & HIWORD) | (((long)neighbor) & LOWORD);
	}
	
	/** a convenience function to unpack the long edge neighbor variable into the neighbor index
	 *  and edge length.
	 *  If you only need one of these, then it's probably best to just use that in the function.
	 * @param edge
	 */
	final static private void unpackNeighbor(long edge)
	{
		unpackedLength   = (int)((edge >> 32) & LOWORD);
		unpackedNeighbor = (int)((edge) & LOWORD);
	}
	/**
	 * The entire graph structure is stored here, including all node data and link data.
	 */
	TIntArrayList  parents;
	TByteArrayList intensity;
	
	TByteArrayList classIndex;
	
	TIntArrayList  mass;
	/** for the links in the graph, we use array lists of int arrays lists.  
	 */
	ArrayList<TIntArrayList> children;
	/** for the links in the graph, we use array lists of int arrays lists.  
	 * For the neighbors, I am switching to using array lists of long array lists because on 
	 *  each edge in graph we also need to store the length of the boundary (in terms of the voxel 
	 *  level) that each edge represents.
	 * We will store the Boundary Length in the HIWORD and the Neighbor Index in the LOWORD
	 */
	ArrayList<TLongArrayList> neighbors;
	int numberOfNodes = 0;
	
	/**
	 *  The Graph Shifts are stored at the nodes themselves.  A node will maintain the best
	 *   current graph shift (or no graph shift) that it can make by storing both the index
	 *   of the neighboring node and the floating point value of the shift.
	 */
	TIntArrayList   graphShiftIndex;
	
	TFloatArrayList graphShiftWeight;
	
	TFloatArrayList likelihoodValues=null;
	int classNum=9; // these are updated with the allocateMembers function is called giving the number of classes
	float lvfiller[] = new float[9];
	
	public ThinGraph()
	{
	}
	public ThinGraph(int n,int m,int c)
	{
		allocateMembers(n,m,c);
	}
	/* when classNum is given, we allocate the likelihood cache value with this many entries */
	public ThinGraph(int n, int m, int c, int classNum)
	{
		allocateMembers(n,m,c);
		this.classNum=classNum;
	    this.lvfiller = new float[classNum];
	}
	/** Edges are stored only through the neighbors arrays.
	 * This will store only the neighbor index (as the low-word in a long) and give a 0 boundary length count.
	 * There is another methods that can store the boundary length as well.
	 * 
	 * @see addEdge(int s, int t, int len)
	 */
	public void addEdge(int s, int t)
	{
		neighbors.get(s).add((long)t);  // casting should be implicit here...
		neighbors.get(t).add((long)s);
	}
	
	public void addEdge(int s, int t, int len)
	{
		neighbors.get(s).add(packNeighbor(len,t));
		neighbors.get(t).add(packNeighbor(len,s));
	}
	
	/**
	 * This is a more clean way of adding a node.  This is a factory model (basically).  The graph will
	 *  instantiate the node, add it internally, and then return the new node.
	 * This hides the ridiculous approach to addNode above away from the user...
	 */
	public ThinGraphNode addNewNode()
	{
		ThinGraphNode TGN = new ThinGraphNode(this,-1);
		addNode(TGN);
		return TGN;
	}
	
	/** This is quite a funky way of going about adding a node, but here, the GraphNodeModel
	 * is assumed to be a ThinGraphNode object that is basically empty (no id).  Once the ThinGraph
	 * allocates the necessary space for the new objects, it will populate the node's id with the
	 * actual id of the node...
	 */
	public void addNode(GraphNodeModel node)
	{
		try
		{
			ThinGraphNode TGN = ThinGraphNode.class.cast(node);
			TGN.setId(numberOfNodes++);
			parents.add(-1);
			mass.add(1);
			intensity.add((byte)0);
			classIndex.add((byte)255);
			neighbors.add(new TLongArrayList(2));
			if (children != null)
				children.add(new TIntArrayList(2));
			if (likelihoodValues != null)
				likelihoodValues.add(lvfiller);
			graphShiftIndex.add(-1);
			graphShiftWeight.add(0);
		}
		catch (ClassCastException e)
		{
			System.out.println("ERR: ThinGraph expects ThinGraphNodes to be added. Added a node, and disregarded the input.");
		}
	}
	

	/** allocate the storage for the graph.  Do deep allocation of child and neighbor
	 *   lists as well
	 *  This does not set the number of nodes.  It only allocates memory for the graph.
	 * @param n  Number of nodes in the graph (initially).
	 * @param m  Number of neighbors per node (initially).
	 * @param c  Number of children per node (initially).  If c is 0, then no children
	 *            storage is allocated at all.
	 */
	public void allocateMembers(int n,int m,int c)
	{
		parents    = new TIntArrayList(n);
		mass       = new TIntArrayList(n);
		intensity  = new TByteArrayList(n);
		classIndex = new TByteArrayList(n);
		neighbors  = new ArrayList<TLongArrayList>(n);
		for (int i=0;i<n;i++)
		{
			TLongArrayList T = new TLongArrayList(m);
			neighbors.add(T);
		}
		if (c > 0)
		{
			likelihoodValues = new TFloatArrayList(n*classNum);
			children   = new ArrayList<TIntArrayList>(n);
			for (int i=0;i<n;i++)
			{
				TIntArrayList T = new TIntArrayList(c);
				children.add(T);
			}
		}
		graphShiftIndex  = new TIntArrayList(n);
		graphShiftWeight = new TFloatArrayList(n);
	}

	public TByteArrayList getClassIndexBuffer()
	{
		return classIndex;
	}
	
	public GraphEdgeModel getEdge(int i)
	{
		throw new AssertionError("Rethink whether you actually need the edge (if so, then rewrite this class).");
	}
	
	public Iterator<GraphEdgeModel> getEdgeIterator()
	{
		throw new AssertionError("Rethink whether or not you actually need the edge iterator");
	}
	
	
	/** 
	 * I simply instantiate a new ThinGraphNode each time this is called.  I looked into
	 * using a WeakHashMap or an array of WeakReferences, but since the graph nodes take up so little
	 * memory and will be used for such a short period of time, I don't think those classes are worthwhile here.
	 */
	public GraphNodeModel getNode(int i)
	{
		return new ThinGraphNode(this,i);
	}
	
	public Iterator<GraphNodeModel> getNodeIterator()
	{
		return new ThinGraphNodeIterator(this);
	}

	public int getNumberOfEdges()
	{
		return -1; // Eh, why bother.
	}

	public int getNumberOfNodes()
	{
		return numberOfNodes;
	}

	public void NDaddChild(int id, int cid)
	{
		if (children != null)
		{
			children.get(id).add(cid);
		}
	}
	
    /////////////////////////////////////////////////////////////////////////////////////	
	//  Node delegate methods 
	
	public void NDaddToBoundaryLength(int id, int nid, int len)
	{
		long mod = packNeighbor(len, 0);
		neighbors.get(id).set(nid, neighbors.get(id).get(nid) + mod);
	}
	
	public void NDaddToLikelihood(int id, int c,float v)
	{
		int i = id*classNum+c;
		likelihoodValues.set(i,likelihoodValues.get(i)+v);
	}
	
	final public int NDgetBoundaryLength(int id, int nid)
	{
		return (int)(((neighbors.get(id).get(nid)) >> 32) & LOWORD);
	}
	
	
	/** We can only return an integer from this class at this time.  If it is
	 * determined that we need to actually return the GraphNodeModel object to 
	 * conform to a future protocol, then we will need to store the child-graph
	 * as well.
	 * @param i
	 * @return
	 */
	public int NDgetChild(int id, int cid)
	{
		if (children != null)
		{
			return children.get(id).get(cid);
		}
		return -1;
	}
	
	public int NDgetClassIndex(int id)
	{
		return classIndex.get(id)&0xFF;
	}
	
	public int NDgetGraphShiftIndex(int id)
	{
		return graphShiftIndex.get(id);
	}
	public float NDgetGraphShiftWeight(int id)
	{
		return graphShiftWeight.get(id);
	}
	
	public byte NDgetIntensity(int id)
	{
		return intensity.get(id);
	}
	
	public int NDgetIntensityUnsigned(int id)
	{
		return 0xFF & (intensity.get(id));
	}
	
	public float NDgetLikelihood(int id, int c)
	{
		return likelihoodValues.get(id*classNum+c);
	}
	
	public int NDgetMass(int id)
	{
		return mass.get(id);
	}
	
	public int NDgetNeighbor(int id, int nid)
	{
		return (int)((neighbors.get(id).get(nid)) & LOWORD);
	}
	
	public GraphNodeModel NDgetNeighborNode(int id, int nid)
	{
		return getNode((int)((neighbors.get(id).get(nid)) & LOWORD));
	}
	
	public int NDgetNumberOfChildren(int id)
	{
		if (children != null)
		{
			return children.get(id).size();
		}
		return 0;
	}
	
	public int NDgetNumberOfNeighbors(int id)
	{
		return neighbors.get(id).size();
	}
	
	public int NDgetParent(int id)
	{
		return parents.get(id);
	}
	
	/**
	 * This function takes the actual index of the neighbor in the graph and does a 
	 * reverse lookup for that node index in the node id's neighbor list
	 * @param id
	 * @param nid  the index in the graph array (not the neighbor array)
	 * @return the index in the neighbor array  (-1 if not in the neighbor array)
	 */
	public int NDlookupNeighborIndex(int id, int nid)
	{
		int v=-1;
		
		int n = neighbors.get(id).size();
		for (int i=0;i<n;i++)
		{
			if(nid == (int)((neighbors.get(id).get(i)) & LOWORD))
			{
				v = i;
				break;
			}
		}
		
		return v;
	}
	
	/**
	 * This removes a child of node id.  The child it removes is given by the index into the 
	 *  id's children array, and not the child id.
	 * @param id
	 * @param childId  Offset of child node in id's children list.
	 */
	public void NDremoveChild(int id, int childId)
	{
		children.get(id).remove(childId);
	}
	
	/**
	 * This removes a child of node id.  The child it removes is given by the actual id of the child.
	 * This method will search through the children of id list to find which index is actual the childId.
	 * So, if possible call NDremoveChild and give the offset already
	 * @param id  Index of the node.
	 * @param childId  Id of the child node.
	 */
	public void NDremoveChildId(int id, int childId)
	{
		int cidx = children.get(id).indexOf(childId);
		if (id != -1)
			children.get(id).remove(cidx);
	}
	
	
	/**
	 * Set the length fo the boundary for a neighbor.
	 * @param id
	 * @param nid  This is the index into the id's neighbor list and not the neighbors index in the graph.
	 * @param len
	 */
	final public void NDsetBoundaryLength(int id, int nid, int len)
	{
		long current = neighbors.get(id).get(nid);
		long updated = ((((long)len) << 32) & HIWORD) | (current & LOWORD);
		neighbors.get(id).set(nid,updated);
	}
	
	public void NDsetClassIndex(int id, int cidx)
	{
		classIndex.set(id,(byte)cidx);
	}
	
	
	public void NDsetGraphShift(int id, int sid, float weight)
	{
		graphShiftIndex.set(id,sid);
		graphShiftWeight.set(id,weight);
	}
	
	public void NDsetGraphShiftIndex(int id, int sid)
	{
		graphShiftIndex.set(id,sid);
	}

	
	public void NDsetGraphShiftWeight(int id, float weight)
	{
		graphShiftWeight.set(id,weight);
	}
						
	
	public void NDsetIntensity(int id, byte v)
	{
		intensity.set(id,v);
	}
	
	public void NDsetLikelihood(int id, int c,float v)
	{
		likelihoodValues.set(id*classNum+c,v);
	}
	
	
	public void NDsetMass(int id, int v)
	{
		mass.set(id,v);
	}
	public void NDsetParent(int id, int v)
	{
		parents.set(id,v);
	}
	
	final public void NDsubtractFromBoundaryLength(int id, int nid, int len)
	{
		long mod = packNeighbor(len, 0);
		neighbors.get(id).set(nid, neighbors.get(id).get(nid) - mod);
	}
	
	public void NDsubtractFromLikelihood(int id, int c,float v)
	{
		int i = id*classNum+c;
		likelihoodValues.set(i,likelihoodValues.get(i)-v);
	}
	protected void populateWithDefault(int n)
	{
		this.classIndex.fill(0,n,(byte)255);
		this.mass.fill(0,n,1);
		this.parents.fill(0,n,-1);
		this.intensity.fill(0,n,(byte)0);
		this.graphShiftIndex.fill(0,n,-1);
		this.graphShiftWeight.fill(0,n,0);
		if (this.likelihoodValues != null)
			this.likelihoodValues.fill(0,n*classNum,0);
	}
	
	public void printDebugInformation()
	{
		System.out.printf("ThinGraph Printing Debug Information\n");
		System.out.printf("Node Number is %d\n",numberOfNodes);
		System.out.printf("length of parents is %d\n",parents.size());
		System.out.printf("length of intensity is %d\n",intensity.size());
		System.out.printf("length of mass %d\n",mass.size());
		System.out.printf("length of classIndex %d\n",classIndex.size());
		System.out.printf("length of neighbors %d\n",neighbors.size());
		if (children == null)
		{	
			System.out.printf("children is null\n");
		}
		else
		{
			System.out.printf("length of children is %d\n",children.size());
		}
	}
	
	public int printNeighbors(int id)
	{
		int v=-1;
		
		int n = neighbors.get(id).size();
		System.out.printf("Node %d has %d neighbors: ",id,n);
		for (int i=0;i<n;i++)
			System.out.printf("%d(%d) ",(int)((neighbors.get(id).get(i)) & LOWORD),
		                                    (((neighbors.get(id).get(i)) >> 32) & LOWORD)
					);
		System.out.printf("\n");
		
		return v;
	}
	
	/* Remove the edge between s and t by scanning through each's edge list.
	 */
	public void removeEdge(int s, int t)
	{
		int nid = NDlookupNeighborIndex(s,t);
		if (nid != -1)
			neighbors.get(s).remove(nid);
		nid = NDlookupNeighborIndex(t,s);
		if (nid != -1)
			neighbors.get(t).remove(nid);
	}

}
