/**
 * 
 */
package edu.ucla.ccb.graphshifts.image;

import java.awt.Color;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * The ImageLabelSet stores a set of image labels (ImageLabel) objects and provides the
 *  ability to query them based on index, read/write to the xml storage format, and 
 *  also generate a colormap (for my code) in argb format.
 *  
 *  Unexpected behavior will happen if two labels in a set have the same index...and this is not prevented.
 *  TODO:  prevent it... given ownership of a label to a set and then check if it is possible to setIndex on
 *   a label based on the other indices in the set.
 * @author jcorso
 *
 */
public class ImageLabelSet implements Serializable
{
	HashMap<Integer,ImageLabel> labels;
	String xmlFilePath = null;
	
	public ImageLabelSet()
	{
		labels = new HashMap<Integer,ImageLabel>();
	}
	
	public void add(ImageLabel L)
	{
		labels.put(L.getIndex(),L);
	}
	
	public ImageLabel get(int i)
	{
		return labels.get(i);
	}
	
	public int getSize()
	{
		return labels.size();
	}
	
	public Iterator<ImageLabel> getIterator()
	{
		return labels.values().iterator();
	}
	
	/**
	 * Return a colormap for the imagelabelset
	 * @param alpha  0-255 transparency.
	 * @return
	 */
	public int[] getColormap(int alpha)
	{
		final int RGB_BITMASK = 0x00FFFFFF;
		int maxNum = computeMaxIndex()+1;
		if (maxNum < 1)
		{
			int colormap[] = new int[1];
			colormap[0] = 0;
			return colormap;
		}
			
		int colormap[] = new int[maxNum];
		
		for (int i=0;i<maxNum;i++)
		{
			ImageLabel L = labels.get(i);
			if (L == null)
			{
				colormap[i] = 0;
			}
			else if (L.isInvisible())
			{
				colormap[i] = 0;	
			}
			else
			{
				colormap[i] = (alpha<<24) | (RGB_BITMASK & (L.getColor().getRGB()));
			}
		}
		return colormap;
	}
	public Colormap getColormap2(int alpha)
	{
		int [] map = getColormap(alpha);
		return Colormap.createFromMap(map);
	}
	
	public int computeMaxIndex()
	{
		int L=-1;
		Collection<ImageLabel> C = labels.values();
		Iterator<ImageLabel> it = C.iterator();
		while (it.hasNext())
		{
			ImageLabel il = it.next();
			if (il.getIndex() > L)
				L = il.getIndex();
		}
		return L;
	}
	
	
	
	static public ImageLabelSet createFromXML(String XMLPath)
	{
		File f = new File(XMLPath);
		if (!f.canRead())
			return null;
		
		ImageLabelSet S = new ImageLabelSet();
		S.xmlFilePath = XMLPath;
		
		Document document = null;
		
		DocumentBuilderFactory factory =
			DocumentBuilderFactory.newInstance();
		//factory.setValidating(true);   
		//factory.setNamespaceAware(true);
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.parse( f );
		} catch (SAXException sxe) {
			// Error generated during parsing)
			Exception  x = sxe;
			if (sxe.getException() != null)
				x = sxe.getException();
			x.printStackTrace();
			
		} catch (ParserConfigurationException pce) {
			// Parser with specified options can't be built
			pce.printStackTrace();
			
		} catch (IOException ioe) {
			// I/O error
			ioe.printStackTrace();
		}
		NodeList list;
		list = document.getElementsByTagName("label");
		for (int i=0;i<list.getLength();i++)
		{
			Node set = list.item(i);
			NamedNodeMap setAttr = set.getAttributes();
			
			ImageLabel lab = new ImageLabel();
			lab.setName(setAttr.getNamedItem("name").getTextContent());
			lab.setIndex(Integer.parseInt(setAttr.getNamedItem("index").getTextContent()));
			
			String str = setAttr.getNamedItem("color").getTextContent();
			String cols[] = str.split(",");
			if (cols.length != 3)
				System.err.println("Error while parsing color of image label"+lab.getName());
			lab.setColor(new Color(Integer.parseInt(cols[0]),Integer.parseInt(cols[1]),Integer.parseInt(cols[2])));
			
			Node inv = setAttr.getNamedItem("invisible");
			if (inv != null)
				lab.setInvisible(Boolean.parseBoolean(inv.getTextContent()));
			
			S.add(lab);
		}
		return S;
	}
	
	
	/**
	 * This does not do buffered writing...
	 * @param XMLPath
	 * @throws IOException
	 */
	public void saveAsXML(String XMLPath) throws IOException
	{
		File f = new File(XMLPath);
		f.createNewFile(); 
		if (!f.canWrite())
			throw new IOException("Cannot open XML file for writing? " + f.getAbsolutePath());
		
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(f)));
		out.println("<labelset>");
		
		Iterator<ImageLabel> it = getIterator();
		while (it.hasNext())
			it.next().writeAsXML(out);
		
		out.println("</labelset>");
		out.close();
	}
	
	/**
	 * Create a labelset that has only one entry, which is a transparent black label named
	 *  "Background" with index 0.  By convention, this is the default, initial label set...
	 * @return
	 */
	static public ImageLabelSet createInitialLabelSet()
	{
		ImageLabelSet S = new ImageLabelSet();

		ImageLabel L = new ImageLabel();
		L.setIndex(0);
		L.setColor(Color.BLACK);
		L.setName("Background");
		L.setInvisible(true);
		
		S.add(L);
		
		return S;
	}
}
