package edu.ucla.ccb.graphshifts.image;

import java.io.IOException;

import net.sourceforge.jiu.codecs.CodecMode;
import net.sourceforge.jiu.codecs.InvalidFileStructureException;
import net.sourceforge.jiu.codecs.PNGCodec;
import net.sourceforge.jiu.codecs.UnsupportedTypeException;
import net.sourceforge.jiu.codecs.WrongFileFormatException;
import net.sourceforge.jiu.data.MemoryBilevelImage;
import net.sourceforge.jiu.data.MemoryRGB24Image;
import net.sourceforge.jiu.data.PixelImage;
import net.sourceforge.jiu.ops.MissingParameterException;
import net.sourceforge.jiu.ops.OperationFailedException;

public class ImageOutliner
{

	public enum Orientation {
		XY,
		XZ,
		ZY;
	}


	Image3PixelAccess raw;
	Image3PixelAccess labels = null;
	Colormap cmap;
	int w,h,d;
	MemoryBilevelImage lastEdgeImage = null;
	MemoryRGB24Image lastOutlineImage = null;
	
	
	public ImageOutliner (Image3PixelAccess raw, Colormap cmap)
	{
		this(raw,null,cmap);
	}
	
	public ImageOutliner (Image3PixelAccess raw, Image3PixelAccess labels, Colormap cmap)
	{
		this.raw = raw;
		this.labels = labels;
		this.cmap = cmap;
		
		w = raw.getWidth();
		h = raw.getHeight();
		d = raw.getDepth();
	}
	
	
	private boolean checkBorder2(Image3PixelAccess labels, int label, int x, int y, int z)
	{
		int w,h;
		w = labels.getWidth();
		h = labels.getHeight();
		
		/* now test if any of its neighbors are not in mask */
		boolean isBorder=false;
		borderCheck:
		{
			if ( (x>0) && (labels.getPixelInt(x-1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>1) && (labels.getPixelInt(x-2, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y>0) && (labels.getPixelInt(x, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y>1) && (labels.getPixelInt(x, y-2, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (labels.getPixelInt(x+1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-2) && (labels.getPixelInt(x+2, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y<h-1) && (labels.getPixelInt(x, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y<h-2) && (labels.getPixelInt(x, y+2, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (y>0) && (labels.getPixelInt(x-1, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (y<h-1) && (labels.getPixelInt(x-1, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (y>0) && (labels.getPixelInt(x+1, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (y<h-1) && (labels.getPixelInt(x+1, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
		}
		return isBorder;
	}
	
	
	private boolean checkBorderXY_1(Image3PixelAccess labels, int label, int x, int y, int z)
	{
		int w,h;
		w = labels.getWidth();
		h = labels.getHeight();
		
		/* now test if any of its neighbors are not in mask */
		boolean isBorder=false;
		borderCheck:
		{
			if ( (x>0) && (labels.getPixelInt(x-1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y>0) && (labels.getPixelInt(x, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (labels.getPixelInt(x+1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y<h-1) && (labels.getPixelInt(x, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (y>0) && (labels.getPixelInt(x-1, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (y<h-1) && (labels.getPixelInt(x-1, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (y>0) && (labels.getPixelInt(x+1, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (y<h-1) && (labels.getPixelInt(x+1, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
		}
		return isBorder;
	}
	
	private boolean checkBorderXZ_1(Image3PixelAccess labels, int label, int x, int y, int z)
	{
		int w,d;
		w = labels.getWidth();
		d = labels.getDepth();
		
		/* now test if any of its neighbors are not in mask */
		boolean isBorder=false;
		borderCheck:
		{
			if ( (x>0) && (labels.getPixelInt(x-1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z>0) && (labels.getPixelInt(x, y, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (labels.getPixelInt(x+1, y, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z<d-1) && (labels.getPixelInt(x, y, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (z>0) && (labels.getPixelInt(x-1, y, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x>0) && (z<d-1) && (labels.getPixelInt(x-1, y, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (z>0) && (labels.getPixelInt(x+1, y, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (x<w-1) && (z<d-1) && (labels.getPixelInt(x+1, y, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
		}
		return isBorder;
	}
	private boolean checkBorderZY_1(Image3PixelAccess labels, int label, int x, int y, int z)
	{
		int h,d;
		h = labels.getHeight();
		d = labels.getDepth();
		
		/* now test if any of its neighbors are not in mask */
		boolean isBorder=false;
		borderCheck:
		{
			if ( (z>0) && (labels.getPixelInt(x, y, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y>0) && (labels.getPixelInt(x, y-1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z<d-1) && (labels.getPixelInt(x, y, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (y<h-1) && (labels.getPixelInt(x, y+1, z) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z>0) && (y>0) && (labels.getPixelInt(x, y-1, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z>0) && (y<h-1) && (labels.getPixelInt(x, y+1, z-1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z<d-1) && (y>0) && (labels.getPixelInt(x, y-1, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
			if ( (z<d-1) && (y<h-1) && (labels.getPixelInt(x, y+1, z+1) != label) )
			{
				isBorder=true;
				break borderCheck;
			}
		}
		return isBorder;
	}

	public Image3PixelAccess getLabels()
	{
		return labels;
	}
	
	public MemoryBilevelImage getLastEdgeImage()
	{
		return lastEdgeImage;
	}
	
	
	public MemoryRGB24Image getLastOutlineImage()
	{
		return lastOutlineImage;
	}

	public Image3PixelAccess getRaw()
	{
		return raw;
	}
	
	public PixelImage fill(int slice, Orientation orient, int[] labelList) 
	{
		if (labels == null)
			return null;
		
		MemoryBilevelImage E;
		MemoryRGB24Image I;
		
		if (orient == Orientation.XZ)
		{
			E = new MemoryBilevelImage(w,d);
			I = new MemoryRGB24Image(w,d);
		}
		else if (orient == Orientation.ZY)
		{
			E = new MemoryBilevelImage(d,h);
			I = new MemoryRGB24Image(d,h);
		}
		else // XY is the default
		{
			E = new MemoryBilevelImage(w,h);
			I = new MemoryRGB24Image(w,h);
		}
		
		lastEdgeImage = E;
		lastOutlineImage = I;
			
		E.clear(MemoryBilevelImage.BLACK);
		setWhite(I);
		
		for (int i=0;i<labelList.length;i++)
		{
			int label = labelList[i];
			int color = cmap.getColorAsIntARGB(label);
			
			if (orient == Orientation.XZ)
			{
				int y = slice;
				for (int z=0;z<d;z++)
					for (int x=0;x<w;x++)
					{
						if (E.isWhite(x, z))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, x, z);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							setRGBSample(color,I,x,z);
							E.putWhite(x, z);
						}
					}
			}
			else if (orient == Orientation.ZY)
			{
				int x = slice;
				for (int y=0;y<h;y++)
					for (int z=0;z<d;z++)
					{
						if (E.isWhite(z, y))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, z, y);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							setRGBSample(color,I,z,y);
							E.putWhite(z, y);
						}
					}
			}
			else
			{
				int z = slice;
				for (int y=0;y<h;y++)
					for (int x=0;x<w;x++)
					{
						if (E.isWhite(x, y))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, x, y);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							setRGBSample(color,I,x,y);
							E.putWhite(x, y);
						}
					}
			}
		}
		
		return I;
	}

	public PixelImage outline(int slice, Orientation orient, int[] labelList) 
	{
		if (labels == null)
			return null;
		
		MemoryBilevelImage E;
		MemoryRGB24Image I;
		
		if (orient == Orientation.XZ)
		{
			E = new MemoryBilevelImage(w,d);
			I = new MemoryRGB24Image(w,d);
		}
		else if (orient == Orientation.ZY)
		{
			E = new MemoryBilevelImage(d,h);
			I = new MemoryRGB24Image(d,h);
		}
		else // XY is the default
		{
			E = new MemoryBilevelImage(w,h);
			I = new MemoryRGB24Image(w,h);
		}
		
		lastEdgeImage = E;
		lastOutlineImage = I;
			
		E.clear(MemoryBilevelImage.BLACK);
		setWhite(I);
		
		for (int i=0;i<labelList.length;i++)
		{
			int label = labelList[i];
			int color = cmap.getColorAsIntARGB(label);
			
			if (orient == Orientation.XZ)
			{
				int y = slice;
				for (int z=0;z<d;z++)
					for (int x=0;x<w;x++)
					{
						if (E.isWhite(x, z))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, x, z);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							if (checkBorderXZ_1(labels, label, x, y, z))
							{
								setRGBSample(color,I,x,z);
								E.putWhite(x, z);
							}
						}
					}
			}
			else if (orient == Orientation.ZY)
			{
				int x = slice;
				for (int y=0;y<h;y++)
					for (int z=0;z<d;z++)
					{
						if (E.isWhite(z, y))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, z, y);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							if (checkBorderZY_1(labels, label, x, y, z))
							{
								setRGBSample(color,I,z,y);
								E.putWhite(z, y);
							}
						}
					}
			}
			else
			{
				int z = slice;
				for (int y=0;y<h;y++)
					for (int x=0;x<w;x++)
					{
						if (E.isWhite(x, y))
							continue;

						setGRAYRGBSample(raw.getPixelInt(x,y,z), I, x, y);

						/* first test if the pixel is in the region */
						if (labels.getPixelInt(x, y, z) == label)
						{
							if (checkBorderXY_1(labels, label, x, y, z))
							{
								setRGBSample(color,I,x,y);
								E.putWhite(x, y);
							}
						}
					}
			}
		}
		
		return I;
	}

	private void setGRAYRGBSample(int gray, MemoryRGB24Image RGB, int x, int y)
	{
		RGB.putSample(MemoryRGB24Image.INDEX_RED,x,y,   gray);
		RGB.putSample(MemoryRGB24Image.INDEX_GREEN,x,y, gray);
		RGB.putSample(MemoryRGB24Image.INDEX_BLUE,x,y,  gray);
	}

	public void setLabels(Image3PixelAccess labels)
	{
		this.labels = labels;
	}
	
	public void setRaw(Image3PixelAccess raw)
	{
		this.raw = raw;
	}
	
	private void setRGBSample(int packed, MemoryRGB24Image RGB, int x, int y)
	{
		RGB.putSample(MemoryRGB24Image.INDEX_RED,x,y,   (0x00FF0000 & packed) >> 16);
		RGB.putSample(MemoryRGB24Image.INDEX_GREEN,x,y, (0x0000FF00 & packed) >> 8);
		RGB.putSample(MemoryRGB24Image.INDEX_BLUE,x,y,  (0x000000FF & packed) );
	}
	
	private void setWhite(MemoryRGB24Image I)
	{
		for(int y=0;y<I.getHeight();y++)
			for(int x=0;x<I.getWidth();x++)
				setGRAYRGBSample(255,I,x,y);
	}
	
	private void writeImage(String path, PixelImage image) throws IOException, InvalidFileStructureException, MissingParameterException, UnsupportedTypeException, WrongFileFormatException, OperationFailedException
	{
		PNGCodec codec = new PNGCodec();
		codec.setFile(path, CodecMode.SAVE);
		codec.setImage(image); 
		codec.process();
		codec.close();
	}


	public void writeLastEdgeImage(String path)
	{
		try
		{
			if (lastEdgeImage != null)
			{
				writeImage(path,lastEdgeImage);
			}
		}
		catch (Exception e)
		{
			System.err.println("error while writing the edge image to "+path);
			e.printStackTrace();
		}
	}


	public void writeLastOutlineImage(String path)
	{
		try
		{
			if (lastOutlineImage != null)
			{
				writeImage(path,lastOutlineImage);
			}
		}
		catch (Exception e)
		{
			System.err.println("error while writing the outline image to "+path);
			e.printStackTrace();
		}
	}

}
