package edu.ucla.ccb.graphshifts.image;

import java.awt.Color;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferInt;
import java.awt.image.Raster;
import java.awt.image.SinglePixelPackedSampleModel;
import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Random;
import java.util.Vector;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriter;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.spi.ImageWriterSpi;
import javax.imageio.stream.FileImageInputStream;
import javax.imageio.stream.FileImageOutputStream;
import javax.imageio.stream.ImageOutputStream;

import edu.ucla.ccb.graphshifts.data.Point3I;
import net.sourceforge.jiu.codecs.CodecMode;
import net.sourceforge.jiu.codecs.ImageLoader;
import net.sourceforge.jiu.codecs.PNGCodec;
import net.sourceforge.jiu.data.IntegerImage;
import net.sourceforge.jiu.data.MemoryGray8Image;
import net.sourceforge.jiu.data.PixelImage;
import net.sourceforge.jiu.data.RGB24Image;
import net.sourceforge.jiu.data.RGBIndex;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import gnu.trove.TIntIntHashMap;



/**
 * ImageUtilities provides some static functions that I use repeatedly
 * in my work with images.
 *
 * @author Jason J. Corso
 */
public class ImageUtilities
{
	
	static public boolean check2D3DFilename(String fn)
	{
		int lio = fn.lastIndexOf('.');
		
		boolean useAnalyze=false;
		
		if (lio == -1)
			useAnalyze = true;
		else
		{
			String suffix = fn.substring(lio+1,fn.length());
//			System.out.printf("parsing suffix %s\n",suffix);
			
			if (suffix.equalsIgnoreCase("HDR") || suffix.equalsIgnoreCase("IMG"))
				useAnalyze = true;
		}
		
		if (useAnalyze == false)
		{
			// we need to check one last thing in the case there was a dot somewhere in the filesystem
			File f = new File(fn + ".hdr");
			File f2 = new File(fn + ".img");
//			System.out.printf("Checking %s %b, and %s %b\n",fn+".hdr",f.exists(),fn+".img",f2.exists());
			if ( (f.exists()) || (f2.exists()) )
				useAnalyze = true;
		}
		
//		System.out.println("using a 3D reader for "+fn+" "+useAnalyze);
		
		return useAnalyze;
	}

	
	static public ScalarImageI3 applyIntegerMap(Image3PixelAccess I, HashMap<Integer,Integer> map)
	{
		ScalarImageI3 J = new ScalarImageI3(I.getWidth(),I.getHeight(),I.getDepth());
		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					int v = I.getPixelInt(x,y,z);
					J.setPixelAt(x,y,z,map.get((Integer)v));
				}
		
		return J;
	}
	
	static public ScalarImageB3 applyIntegerMapByte(Image3PixelAccess I, HashMap<Integer,Integer> map)
	{
		ScalarImageB3 J = new ScalarImageB3(I.getWidth(),I.getHeight(),I.getDepth());
		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					int v = I.getPixelInt(x,y,z);
					J.setPixelAt(x,y,z,map.get((Integer)v));
				}
		
		return J;
	}
	
	static public void applyMask(Image3PixelAccess I, Image3PixelAccess M, ScalarImageI3 O, int v)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
					O.setPixelAt(x, y, z, (M.getPixelInt(x, y, z) == 0) ? v : I.getPixelInt(x, y, z));
	}
	
	
	static public void clampI3toB3 (ScalarImageI3 I, ScalarImageB3 B)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
				{
					B.setPixelAt(x, y, z,I.getPixelInt(x,y,z));
				}
		
	}
	
	static public ScalarImageI3 convertF3toI3(ScalarImageF3 F)
	{
		ScalarImageI3 I = new ScalarImageI3(F.getWidth(),
				F.getHeight(),F.getDepth());
		
		for (int z=0;z<F.getDepth();z++)
			for (int y=0;y<F.getHeight();y++)
				for (int x=0;x<F.getWidth();x++)
				{
					I.setPixelAt(x,y,z,(int)F.getPixelValueAt(x,y,z));
				}
		
		return I;
	}
	

	static public BufferedImage createRedGreenChecks(int w, int h, float a)
	{
		int[] buf = new int[w*h];
		final int s = 10;
		int aa = (int)(a*255.0f);
		int red = (aa<<24)|(0x00ff0000);
		int green = (aa<<24)|(0x0000ff00);

		for(int y=0;y<h;y++)
		{
			int yt = (y/s)&(0x1);
			for(int x=0;x<w;x++)
			{
				if( (yt ^ ((x/s)&(0x1))) == 1)
					buf[y*w+x] = red;	
				else
					buf[y*w+x] = green;	
			}
		}

		DataBufferInt dbi = new DataBufferInt(buf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
			(DataBuffer.TYPE_INT, w,h,w,
		  new int[] {0x00FF0000,0x0000FF00,0x000000FF,0xFF000000});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
	
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_INT_ARGB);
		I.setData(r);
		return I;
	}
	
	
	
	static public Image3PixelAccess createMaskFromSet(Image3PixelAccess I[], int value)
	{
		ScalarImageB3 mask = new ScalarImageB3(I[0].getWidth(),I[0].getHeight(),I[0].getDepth());
		for (int z=0;z<I[0].getDepth();z++)
			for (int y=0;y<I[0].getHeight();y++)
				for (int x=0;x<I[0].getWidth();x++)
				{
					int v = I[0].getPixelInt(x, y, z);
					boolean maskIn=true;
					for (int i=0;i<I.length&&maskIn;i++)
						if (v==value)
							maskIn=false;
					
					if (maskIn)
						mask.setPixelAt(x, y, z, 1);
					else 
						mask.setPixelAt(x, y, z, 0);
				}
		return mask;
	}
	
	
	static public Image3PixelAccess crop_B3(Image3PixelAccess I, int xc, int yc, int zc)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		int ww,hh,dd;
		ww = w - 2 * xc;
		hh = h - 2 * yc;
		dd = d - 2 * zc;
		
		ScalarImageB3 O = new ScalarImageB3(ww,hh,dd);
		
		int x,y,z,xx,yy,zz;
		for (z=zc,zz=0;zz<dd;z++,zz++)
			for (y=yc,yy=0;yy<hh;y++,yy++)
				for (x=xc,xx=0;xx<ww;x++,xx++)
				{
					O.setPixelAt(xx, yy, zz, I.getPixelByte(x, y, z));
				}
		
		return O;
	}
	
	static public IntegerImage crop_2D_Gray(IntegerImage I, int xc, int yc)
	{
		int w,h;
		w = I.getWidth();
		h = I.getHeight();
		
		int ww,hh;
		ww = w - 2 * xc;
		hh = h - 2 * yc;
		
		MemoryGray8Image O = new MemoryGray8Image(ww,hh);
		
		int x,y,xx,yy;
		for (y=yc,yy=0;yy<hh;y++,yy++)
			for (x=xc,xx=0;xx<ww;x++,xx++)
			{
				O.putSample(xx, yy, I.getSample(x, y));
			}
		
		return O;
	}
	
	static public Point3I getVoxelCoordsForIndex(int i, Image3Protocol I)
	{
		Point3I P = new Point3I();
		/** these three lines are an example of what happens when you code very tired...
		P.setZ(i % (I.getWidth()*I.getHeight()));
		P.setY((i-P.getZ()*I.getWidth()*I.getHeight()) % I.getWidth());
		P.setX((i-(P.getZ()*I.getWidth()*I.getHeight())-P.getY()*I.getWidth()));
		*/
		P.setZ((int)Math.floor((double)i / (double)(I.getWidth()*I.getHeight())));
		P.setY((int)Math.floor((double)(i-P.getZ()*I.getWidth()*I.getHeight()) / (double)I.getWidth()));
		P.setX((i-P.getZ()*I.getWidth()*I.getHeight()) % I.getWidth());
		//System.out.println("index "+i+" is "+P.getX()+","+P.getY()+","+P.getZ()+" for w,h of "+I.getWidth()+","+I.getHeight()+","+I.getDepth());
		return P;
	}
	
	
	/** A non-linear rank (median) filter 5x5x5...slow implementation. */
	static public ScalarImageB3 filterMedian5(ScalarImageB3 I)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		ScalarImageB3 O = new ScalarImageB3(w,h,d);
		int[] vals = new int[125]; 
		
		for (int z=2;z<d-2;z++)
			for (int y=2;y<h-2;y++)
				for (int x=2;x<w-2;x++)
				{
					int i=0;
					for (int zz=z-2;zz<=z+2;zz++)
						for (int yy=y-2;yy<=y+2;yy++)
							for (int xx=x-2;xx<=x+2;xx++,i++)
							{
								vals[i] = I.getPixelInt(xx, yy, zz);
							}
					Arrays.sort(vals);
					O.setPixelAt(x, y, z, vals[63]);
				}
		
		return O;
	}
	
	static public ScalarImageB3 filterMin(ScalarImageB3 I, int hw)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		ScalarImageB3 O = new ScalarImageB3(w,h,d);
		
		for (int z=hw;z<d-hw;z++)
			for (int y=hw;y<h-hw;y++)
				for (int x=hw;x<w-hw;x++)
				{
					int i=0;
					int minv=256;
					for (int zz=z-hw;zz<=z+hw;zz++)
						for (int yy=y-hw;yy<=y+hw;yy++)
							for (int xx=x-hw;xx<=x+hw;xx++,i++)
							{
								int v = I.getPixelInt(xx, yy, zz);
								minv = (v < minv) ? v : minv;
							}
					O.setPixelAt(x, y, z, minv);
				}
		
		return O;
	}
	
	static public ScalarImageB3 filterMin5(ScalarImageB3 I)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		ScalarImageB3 O = new ScalarImageB3(w,h,d);
		
		for (int z=2;z<d-2;z++)
			for (int y=2;y<h-2;y++)
				for (int x=2;x<w-2;x++)
				{
					int i=0;
					int minv=256;
					for (int zz=z-2;zz<=z+2;zz++)
						for (int yy=y-2;yy<=y+2;yy++)
							for (int xx=x-2;xx<=x+2;xx++,i++)
							{
								int v = I.getPixelInt(xx, yy, zz);
								minv = (v < minv) ? v : minv;
							}
					O.setPixelAt(x, y, z, minv);
				}
		
		return O;
	}
	
	static public ScalarImageB3 filterMin3(ScalarImageB3 I)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		ScalarImageB3 O = new ScalarImageB3(w,h,d);
		
		for (int z=1;z<d-1;z++)
			for (int y=1;y<h-1;y++)
				for (int x=1;x<w-1;x++)
				{
					int i=0;
					int minv=256;
					for (int zz=z-1;zz<=z+1;zz++)
						for (int yy=y-1;yy<=y+1;yy++)
							for (int xx=x-1;xx<=x+1;xx++,i++)
							{
								int v = I.getPixelInt(xx, yy, zz);
								minv = (v < minv) ? v : minv;
							}
					O.setPixelAt(x, y, z, minv);
				}
		
		return O;
	}

	
	static  public Color intToColor(int clr)
	{
		float r,g,b,a;
		r = ((float)((clr&0x00ff0000) >> 16))/255.f;
		g = ((float)((clr&0x0000ff00) >> 8))/255.f;
		b = ((float)(clr&0x000000ff))/255.f;
		a = ((float)((clr&0xff000000) >> 24))/255.f;
		return new Color(r,g,b,a);
	}
	
	
/*
	static public ScalarImageF3 readAnalyzeViaITK(String filename)
	{
		itkImageFileReaderF3_Pointer reader = 
			    itkImageFileReaderF3.itkImageFileReaderF3_New();
		reader.SetFileName(filename);
		reader.Update();

		itkImageF3 image = reader.GetOutput();

		itkImageRegion3 bounds = image.GetLargestPossibleRegion();
		itkSize3 size = bounds.GetSize();

		ScalarImageF3 i3d = new ScalarImageF3((int)size.GetElement(0),
				          (int)size.GetElement(1),(int)size.GetElement(2));
		itkIndex3 idx = new itkIndex3();

		for(int z=0;z<size.GetElement(2);z++)
		{
			idx.SetElement(2,z);
			for(int y=0;y<size.GetElement(1);y++)
			{
				idx.SetElement(1,y);
				for(int x=0;x<size.GetElement(0);x++)
				{
					idx.SetElement(0,x);
					i3d.setPixelAt(x,y,z,image.GetPixel(idx));	
				}
			}
		}

		return i3d;
	}
*/
	
	
	/**
	 * Compute the cumulative image and pad the first row/column/band with zeros.
	 */
	static public ScalarImageI3 makeCumulativeImageI(Image3PixelAccess I)
	{
		int w,h,d;
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		ScalarImageI3 O = new ScalarImageI3(w+1,h+1,d+1);
		
		Arrays.fill(O.getBuffer().getData(),0);
		
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
			{
				int last = 0;
				for (int x=0;x<w;x++)
				{
					last += I.getPixelInt(x,y,z);
					O.setPixelAt(x+1, y+1, z+1, last);
				}
			}
			
		for (int z=1;z<=d;z++)
			for (int x=1;x<=w;x++)
			{
				int last = 0;
				for (int y=1;y<=h;y++)
				{
					last += O.getPixelInt(x,y,z);
					O.setPixelAt(x, y, z, last);
				}
			}
		
		for (int y=1;y<=h;y++)
			for (int x=1;x<=w;x++)
			{
				int last = 0;
				for (int z=1;z<=d;z++)
				{
					last += O.getPixelInt(x,y,z);
					O.setPixelAt(x, y, z, last);
				}
			}
		
		return O;
	}
	
	static public IntegerImage makeCumulativeImage(IntegerImage I)
	{
		int w,h;
		w = I.getWidth();
		h = I.getHeight();
		
		MemoryGray32Image O = new MemoryGray32Image(w+1,h+1);
		O.clear(0);
		
		for (int y=0;y<h;y++)
		{
			int last = 0;
			for (int x=0;x<w;x++)
			{
				last += I.getSample(x,y);
				O.putSample(x+1, y+1, last);
			}
		}
		
		for (int x=1;x<=w;x++)
		{
			int last = 0;
			for (int y=1;y<=h;y++)
			{
				last += O.getSample(x,y);
				O.putSample(x, y, last);
			}
		}
		
		return O;
	}
	
	static public IntegerImage[] makeCumulativeImage(RGB24Image I)
	{
		int w,h;
		w = I.getWidth();
		h = I.getHeight();
		
		IntegerImage out[] = new IntegerImage[3];
		
		MemoryGray32Image Or = new MemoryGray32Image(w+1,h+1);
		MemoryGray32Image Og = new MemoryGray32Image(w+1,h+1);
		MemoryGray32Image Ob = new MemoryGray32Image(w+1,h+1);
		Or.clear(0);
		Og.clear(0);
		Ob.clear(0);
		
		for (int y=0;y<h;y++)
		{
			int lastr = 0;
			int lastg = 0;
			int lastb = 0;
			for (int x=0;x<w;x++)
			{
				lastr += I.getSample(RGBIndex.INDEX_RED,x,y);
				lastg += I.getSample(RGBIndex.INDEX_GREEN,x,y);
				lastb += I.getSample(RGBIndex.INDEX_BLUE,x,y);
				Or.putSample(x+1, y+1, lastr);
				Og.putSample(x+1, y+1, lastg);
				Ob.putSample(x+1, y+1, lastb);
			}
		}
		
		for (int x=1;x<=w;x++)
		{
			int lastr = 0;
			int lastg = 0;
			int lastb = 0;
			for (int y=1;y<=h;y++)
			{
				lastr += Or.getSample(x,y);
				Or.putSample(x, y, lastr);
				lastg += Og.getSample(x,y);
				Og.putSample(x, y, lastg);
				lastb += Ob.getSample(x,y);
				Ob.putSample(x, y, lastb);
			}
		}
		
		out[0] = Or;
		out[1] = Og;
		out[2] = Ob;
		
		return out;
	}
	
	
	/**
	 * Generate a random colormap for n entries stored as packed integers.
	 * @param alpha The transparency of the pixel value.
	 */
	static public int[] makeRandomizedColorMap(int n, int alpha)
	{
		int colormap[];
		
		Random rand = new Random();
		colormap = new int[n];
		int r,g,b;
		for(int i=0;i<n;i++)
		{
			r = rand.nextInt(255);
			g = rand.nextInt(255);
			b = rand.nextInt(255);
			//pack these bytes into colormap_[i]
			colormap[i] = ((alpha<<24) | (r<<16) | (g<<8) | b);
		}
		
		return colormap;
	}
	
	
	static public Image3PixelAccess readViaJIUTo3D(String filename)
	{
		try
		{
			System.out.println(filename);
			
			PixelImage pi = ImageLoader.load(filename);
			if (pi instanceof IntegerImage)
				return new JIUIntegerImage_Image3PixelAccess_Adaptor((IntegerImage)pi);
			
			throw new AssertionError("Loaded image is not IntegerImage??? FATAL\n");
		}
		catch (Exception e)
		{
			System.out.println("Exception when reading "+filename);
			e.printStackTrace();
		}
		return null;
	}

	
	
	
	static public ScalarImageF3 readViaImageIO(String filename)
	{
		try
		{
			// Test the source file existence
			File srcFile = new File(filename);
			if ( !srcFile.exists() || !srcFile.isFile() || !srcFile.canRead() ) {
				System.out.println("Cannot read \"" + filename + "\"");
				System.exit(1);
			}
			return readViaImageIO(srcFile);
		} 
		catch (Exception e) 
		{
			e.printStackTrace();
			System.exit(-1);
		}
	    return null;
	}
		
	static public ScalarImageF3 readViaImageIO(File srcFile)
	{
		ScalarImageF3 I = null;
		
		try
		{
			if ( !srcFile.exists() || !srcFile.isFile() || !srcFile.canRead() ) {
				System.out.println("Cannot read \"" + srcFile.getPath() + "\"");
				System.exit(1);
			}
			
			// Find an Image Reader for the source file
			ImageReader imageReader = _getImageReader(srcFile);
			if (imageReader == null) {
				System.out.println("Cannot decode \"" + srcFile.getPath() + "\"");
				System.exit(1);
			}
			
			// If available, read the stream metadata
//			IIOMetadata srcMetadata = imageReader.getStreamMetadata();
//			if (srcMetadata != null) {
//				String formatName = srcMetadata.getNativeMetadataFormatName();
//				Node root = srcMetadata.getAsTree(formatName);
//				
//				// YOU CAN CHANGE THE STREAM METADATA HERE --> root
//				displayMetadata(root);
//				displayMetadata(srcMetadata.getAsTree("javax_imageio_1.0"));
//			}
//			
			int w,h,d;
			w = imageReader.getWidth(0);
			h = imageReader.getHeight(0);
			d = imageReader.getNumImages(true);
			
			System.out.println(w+","+h+","+d);
			
			I = new ScalarImageF3(w,h,d);
			float Idata[] = I.getBuffer();
			float Ibuf[] = new float[w*h];
			
			// Read/write the image data
			for (int i = 0; i < d; i++) 
			{
				
				 // If available, read the image metadata
//				IIOMetadata srcMetadata = imageReader.getImageMetadata(i);
//				if (srcMetadata != null) {
//				  String formatName = srcMetadata.getNativeMetadataFormatName();
//				  Node root = srcMetadata.getAsTree(formatName);
//				  displayMetadata(root);
//				}
//				else
//					System.out.println("the image metadata is null, here");
				   
				  
				// Read/write the image data and image metadata
				BufferedImage srcImage = imageReader.read(i);
				Raster r = srcImage.getData();		
		
				r.getSamples(0,0,w,h,0,Ibuf);
				System.arraycopy(Ibuf,0,Idata,i*w*h,w*h);
				
//				 go through the pixels in the BufferedImage and copy to I
//				
//				for (int y=0;y<h;y++)
//					for (int x=0;x<w;x++)
//					{		
//					I.setPixelAt(x,y,i,(float)(r.getSampleFloat(x,y,0)));
//					}

			}
		} 
		catch (Exception e) 
		{
			e.printStackTrace();
			System.exit(-1);
		}
		
		return I;
	}

	
	static public ScalarImageB3 readViaImageIO_B3(String filename)
	{
		try
		{
			// Test the source file existence
			File srcFile = new File(filename);
			if ( !srcFile.exists() || !srcFile.isFile() || !srcFile.canRead() ) {
				System.out.println("Cannot read \"" + filename + "\"");
				System.exit(1);
			}
			return readViaImageIO_B3(srcFile);
		} 
		catch (Exception e) 
		{
			e.printStackTrace();
			System.exit(-1);
		}
	    return null;
	}
		
	static public ScalarImageB3 readViaImageIO_B3(File srcFile)
	{
		/* initially we read the image as a float image and then 
		 * get the scale of the image and set it in the final byte image.
		 * this is a short waste of time, but who said image loading needs
		 * to be fast anyway?
		 */
		ScalarImageF3 F = readViaImageIO(srcFile);
		
		ScalarImageB3 I = new ScalarImageB3(F.getWidth(),F.getHeight(),F.getDepth());
		
		/* NO 
		 * we will linear scale the image pixels to cover the full range of the display
		 * which is 0 - 255
		 *
		float maxvalue = F.getMaxValue();
		float minvalue = F.getMinValue();
		I.setScaledIntensity(minvalue,maxvalue);
		float scale = 255.0f/(maxvalue-minvalue);
		*/
		
		for (int z=0;z<F.getDepth();z++)
			for (int y=0;y<F.getHeight();y++)
				for (int x=0;x<F.getWidth();x++)
				{
//					float f = scale * (F.getPixelFloat(x, y, z) - minvalue);
					I.setPixelAt(x, y, z, (byte)(255.f*F.getPixelFloat(x,y,z)));
//					System.out.printf("F3 is %f and I is now %d\n",F.getPixelFloat(x,y,z),I.getPixelInt(x,y,z));
				}
		
		return I;
	}
	
	
	
	/**
	 * This function will remap the values of a ScalarImageB3 (as an Image3PixelAccess) based onthe
	 *   string parameter. 
	 *   
	 * The string parameter should be comma-separated list of oldvalue1-newvalue1,oldvalue2-newvalue2,...
	 * @param image
	 * @param str
	 */
	static public void remapBytes(Image3PixelAccess image, String str)
	{
		if (!(image instanceof ScalarImageB3))
			throw new AssertionError("image must be uchar\n");
		ScalarImageB3 I = (ScalarImageB3)image;
		
		if (str.equals("0"))
			return;
		
		TIntIntHashMap map = new TIntIntHashMap();
		String[] vals = str.split(",");
		for (int i=0;i<vals.length;i++)
		{
			String[] m = vals[i].split("-");
			int a,b;
			a = Integer.parseInt(m[0]);
			b = Integer.parseInt(m[1]);
			map.put(a, b);
		}
		
		int w,h,d;
		w = image.getWidth();
		h = image.getHeight();
		d = image.getDepth();
		for (int z=0;z<d;z++)
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
				{
					int v = I.getPixelInt(x, y, z);
					if (map.contains(v))
						I.setPixelAt(x, y, z, map.get(v));
				}
	}
	
	
	
	static public void writeGrayViaImageIO(byte[] buf, int w, int h, String fn, String ftype)
	{		
		
		DataBufferByte dbi = new DataBufferByte(buf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
		(DataBuffer.TYPE_BYTE, w,h,w,new int[] {0xFF});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
		
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_BYTE_GRAY);
		I.setData(r);
		try
		{
			ImageIO.write(I, ftype, new File(fn));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
	}

	static public void writePGM(byte[] buf, int w, int h, String fn)
	{
		try
		{
			FileOutputStream fos = new FileOutputStream(fn);
			FileWriter fw = new FileWriter(fos.getFD()); // char-stream
			DataOutputStream dos = new DataOutputStream(fos); // byte-stream
			
			PrintWriter pw = new PrintWriter(fw);
			pw.printf("P5\n%d %d\n255\n", w,h);

			dos.write(buf,0,w*h);
			fos.close();
		}
		catch (IOException e)
		{
			System.err.println("IOException while trying to writePGM");
		}

	}
	
	
	static public void writeDelIDataForLabel_F(Image3PixelAccess I, 
											   Image3PixelAccess L, 
			                                   Image3PixelAccess M, 
			                                   int l, String fn)
	{
		PrintStream out = null;
		try {
			out = new PrintStream(new BufferedOutputStream(new FileOutputStream(fn)));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} 

		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					if ((M.getPixelInt(x, y, z) != 0) && (L.getPixelInt(x, y, z) == l))
					{
						int xx,yy,zz;
						
						// now, we check each of the neighbors
						xx = x-1; yy = y; zz = z;
						if ( (xx > 0) && (M.getPixelInt(xx, yy, zz) != 0) &&
								         (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
						
						xx = x+1; yy = y; zz = z;
						if ( (xx < I.getWidth()) && 
						     (M.getPixelInt(xx, yy, zz) != 0) &&
						     (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
						
						xx = x; yy = y-1; zz = z;
						if ( (yy > 0) && 
						     (M.getPixelInt(xx, yy, zz) != 0) &&
						     (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
						
						xx = x; yy = y+1; zz = z;
						if ( (yy < I.getHeight()) && 
						     (M.getPixelInt(xx, yy, zz) != 0) &&
						     (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
						
						xx = x; yy = y; zz = z-1;
						if ( (zz > 0) && 
						     (M.getPixelInt(xx, yy, zz) != 0) &&
						     (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
						
						xx = x; yy = y; zz = z+1;
						if ( (zz < I.getDepth()) && 
						     (M.getPixelInt(xx, yy, zz) != 0) &&
						     (L.getPixelInt(xx, yy, zz) == l) )
							out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
					}  // if x,y,z, is not masked and has the right label...
				}

		out.close();
	}
	
	
	static public void writePixelDataForLabel_F(Image3PixelAccess I, Image3PixelAccess L, 
			                                    int l, String fn)
	{
		PrintStream out = null;
		try {
			out = new PrintStream(new BufferedOutputStream(new FileOutputStream(fn)));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} 
		
		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					if (L.getPixelInt(x, y, z) == l)
						out.printf("%f\n", I.getPixelFloat(x, y, z));
				}
		
		out.close();
	}
	
	static public void writePixelDataForLabel_F(Image3PixelAccess I, 
			                                    Image3PixelAccess L, 
			                                    Image3PixelAccess M, 
			                                    int l, String fn)
	{
		PrintStream out = null;
		try {
			out = new PrintStream(new BufferedOutputStream(new FileOutputStream(fn)));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} 

		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					if ((M.getPixelInt(x, y, z) != 0) && (L.getPixelInt(x, y, z) == l))
						out.printf("%f\n", I.getPixelFloat(x, y, z));
				}

		out.close();
	}
	
	
	/*
	// writes the 3D volume out...
	static public void writeAnalyzeViaImageIO(Image3Protocol image, String fn) throws IOException
	{
		// prepare the actual image for writing
		image.setSliceOrientation(Image3Protocol.SLICE_XY);
		
		String baseFileName;
		if (fn.endsWith("img") || fn.endsWith("hdr"))
		{
			// strip suffix
			int idx = fn.lastIndexOf(".");
			baseFileName = fn.substring(0,idx);
		}
		else
			baseFileName = fn; // else just append to it
		
		ImageWriter imageWriter = _getAnalyzeImageWriter(baseFileName);
		
		// we have to make the darn metadata... for the image
		//  I am not sure about the correct way of using imageIO to write one
		//   of these streams.  So, I am just creating a headerkey and imagedimension
		//   object with some default settings
		HeaderKey hkey = new HeaderKey(348,new byte[]{},new byte[]{},16384,(short)0,(byte)114,(byte)0);
		
		
		// from another image
//		<IMAGE_DIMENSION dim_0="4" dim_1="32" dim_2="32" dim_3="8" dim_4="1" dim_5="0" dim_6="0" 
//			dim_7="0" vox_units="mm" cal_units="" unused1="0" dataType="2" bitpix="8" dim_un0="0" 
//			pixdim_0="0.0" pixdim_1="1.0" pixdim_2="1.0" pixdim_3="1.0" pixdim_4="0.0" 
//			pixdim_5="0.0" pixdim_6="0.0" pixdim_7="0.0" vox_offset="0.0" roi_scale="1.0" 
//			funused1="0.0" funused2="0.0" cal_max="0.0" cal_min="0.0" compressed="0.0" verified="0.0" 
//			glmax="255" glmin="0"/>
		
		
		// the constructor
//		public ImageDimension(short dim0, short dim1, short dim2, short dim3,
//				short dim4, short dim5, short dim6, short dim7,
//				byte[] voxUnits, byte[] calUnits, short unused1,
//				short dataType, short bitPix, short dimUn0,
//				float pixDim0, float pixDim1, float pixDim2,
//				float pixDim3, float pixDim4, float pixDim5,
//				float pixDim6, float pixDim7, float voxOffset,
//				float roiScale, float fUnused1, float fUnused2,
//				float calMax, float calMin, float compressed,
//				float verified, int glMax, int glMin)

		
		ImageDimension dim = new ImageDimension((short)4,(short)image.getWidth(),(short)image.getHeight(),
				(short)image.getDepth(),(short)1,(short)0,(short)0,(short)0,new byte[]{'m','m'},new byte[]{},
				(short)0,(short)2,(short)8,(short)0,
				0.0f,1.0f,1.0f,
				1.0f,0.0f,0.0f,
				0.0f,0.0f,0.0f,
				1.0f,0.0f,0.0f,
				0.0f,0.0f,0.0f,
				0.0f,255,0);
		
		// Data history obbject
		
//		<DATA_HISTORY descrip="" aux_file="" orient="0" originator="00000000000000000000" generated="" 
//			scannum="" patient_id="" exp_date="" exp_time="" hist_un0="" views="0" vols_added="0" 
//				start_field="0" field_skip="0" omax="0" omin="0" smax="0" smin="0"/>
//		  public DataHistory(byte[] descrip, byte[] auxFile, byte orient,
//				     byte[] originator, byte[] generated, byte[] scannum,
//				     byte[] patientId, byte[] expDate, byte[] expTime,
//				     byte[] histUn0, int views, int volsAdded, int startField,
//				     int fieldSkip, int omax, int omin, int smax, int smin)

		DataHistory history = new DataHistory(new byte[]{},new byte[]{},(byte)0,
				new byte[]{},new byte[]{},new byte[]{},
				new byte[]{},new byte[]{},new byte[]{},
				new byte[]{},0,0,0,
				0,0,0,0,0);
		
		AnalyzeHeader head = new AnalyzeHeader(hkey,dim,history);
				
		Node htree = AnalyzeMetadataConversions.toTree(head);
		
		IIOMetadata metaData = imageWriter.getDefaultStreamMetadata(null);
		
		metaData.mergeTree(metaData.getNativeMetadataFormatName(),htree);
		
//		System.out.println(" metaData format names ");
//		String names[] = metaData.getMetadataFormatNames();
//		for (int i=0;i<names.length;i++)
//			System.out.println("name " + i + " is "+ names[i]);
//		
//		// name 0 is edu_ucla_loni_analyze_1.0
//		// name 1 is javax_imageio_1.0 
//		displayMetadata(metaData.getAsTree("javax_imageio_1.0"));
//		displayMetadata(metaData.getAsTree("edu_ucla_loni_analyze_1.0"));
			
		imageWriter.prepareWriteSequence(metaData);
		
		System.out.println("writign out the image and it has "+image.getDepth()+" slices");
		
		for (int i=0;i<image.getDepth();i++)
		{
//			metaData = imageWriter.getDefaultImageMetadata(null,null);
			BufferedImage I = image.getSlice(i);
//			IIOImage iioI = new IIOImage(I,null,metaData);
			IIOImage iioI = new IIOImage(I,null,null);
			imageWriter.writeToSequence(iioI,null);
		}
		
		imageWriter.endWriteSequence();
		imageWriter.dispose();
	}
	*/
	
	
	static public void writeTwoClassDelIDataForLabel_F(Image3PixelAccess I, 
			Image3PixelAccess L, 
			Image3PixelAccess M, 
			int l1,int l2, String fn)
	{
		PrintStream out = null;
		try {
			out = new PrintStream(new BufferedOutputStream(new FileOutputStream(fn)));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} 

		for (int z=0;z<I.getDepth();z++)
			for (int y=0;y<I.getHeight();y++)
				for (int x=0;x<I.getWidth();x++)
				{
					if (M.getPixelInt(x, y, z) == 0) 
						continue;
					
					int lB;
						
					if (L.getPixelInt(x, y, z) == l1)
					{
						lB = l2;
					}
					else
					if (L.getPixelInt(x, y, z) == l2)
					{
						lB = l1;
					}
					else
						continue;
					
					int xx,yy,zz;
					xx = x+1; yy = y; zz = z;
					if ( (xx < I.getWidth()) && 
							(M.getPixelInt(xx, yy, zz) != 0) &&
							(L.getPixelInt(xx, yy, zz) == lB) )
						out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));

					xx = x; yy = y+1; zz = z;
					if ( (yy < I.getHeight()) && 
							(M.getPixelInt(xx, yy, zz) != 0) &&
							(L.getPixelInt(xx, yy, zz) == lB) )
						out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));

					xx = x; yy = y; zz = z+1;
					if ( (zz < I.getDepth()) && 
							(M.getPixelInt(xx, yy, zz) != 0) &&
							(L.getPixelInt(xx, yy, zz) == lB) )
						out.printf("%f\n",I.getPixelFloat(x, y, z)-I.getPixelFloat(xx,yy,zz));
				}

		out.close();
	}

	
	
	static public void writeViaImageIO_GraytoGenRandomLUT(byte[] buf, int w, int h, String fn, String ftype)
	{
		int ibuf[] = new int[w*h];
		
		// assumes maximum value in the image is the number of entires in the colormap that we need.
		int n=0;
		for (int i=0;i<w*h;i++)
			if (buf[i] > n)
				n = buf[i];
		n++;
		
		int LUT[] = makeRandomizedColorMap(n,255);
		
		for (int i=0;i<w*h;i++)
			ibuf[i] = LUT[buf[i]];
		
		DataBufferInt dbi = new DataBufferInt(ibuf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
			(DataBuffer.TYPE_INT, w,h,w,
		  new int[] {0x00FF0000,0x0000FF00,0x000000FF,0xFF000000});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
	
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_INT_ARGB);
		I.setData(r);
		try
		{
			ImageIO.write(I, ftype, new File(fn));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
	}
	
	static public void writeViaImageIO_GraytoGenRandomLUT(int[] buf, int w, int h, String fn, String ftype)
	{
		int ibuf[] = new int[w*h];
		
		// assumes maximum value in the image is the number of entires in the colormap that we need.
		int n=0;
		for (int i=0;i<w*h;i++)
			if (buf[i] > n)
				n = buf[i];
		n++;
		
		int LUT[] = makeRandomizedColorMap(n,255);
		
		for (int i=0;i<w*h;i++)
			ibuf[i] = LUT[buf[i]];
		
		DataBufferInt dbi = new DataBufferInt(ibuf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
			(DataBuffer.TYPE_INT, w,h,w,
		  new int[] {0x00FF0000,0x0000FF00,0x000000FF,0xFF000000});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
	
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_INT_ARGB);
		I.setData(r);
		try
		{
			ImageIO.write(I, ftype, new File(fn));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
	}
	
	
	static public void writeViaImageIO_Gray2D(ScalarImageF3 I, float scale, String fileprefix, String filetype)
	{
		byte buf[] = new byte[I.getWidth()*I.getHeight()];
		
		for (int y=0;y<I.getHeight();y++)
			for (int x=0;x<I.getWidth();x++)
				buf[y*I.getWidth()+x] = (byte)(I.getPixelValueAt(x,y,0)*scale);
			
		DataBufferByte dbi = new DataBufferByte(buf,I.getWidth()*I.getHeight());
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
		(DataBuffer.TYPE_BYTE, I.getWidth(),I.getHeight(),I.getWidth(),new int[] {0xFF});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
		
		BufferedImage IB = new BufferedImage(I.getWidth(),I.getHeight(), BufferedImage.TYPE_BYTE_GRAY);
		IB.setData(r);
		try
		{
			ImageIO.write(IB, filetype, new File(fileprefix+"."+filetype));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
		
	}

	static public void writeViaImageIO_GraytoColor(byte[] buf, int w, int h, String fn, String ftype, int[] LUT)
	{
		int ibuf[] = new int[w*h];
		
		for (int i=0;i<w*h;i++)
			ibuf[i] = LUT[buf[i]];
		
		DataBufferInt dbi = new DataBufferInt(ibuf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
			(DataBuffer.TYPE_INT, w,h,w,
		  new int[] {0x00FF0000,0x0000FF00,0x000000FF,0xFF000000});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
	
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_INT_ARGB);
		I.setData(r);
		try
		{
			ImageIO.write(I, ftype, new File(fn));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
	}
	
	static public void writeViaImageIO_GraytoColor(int[] buf, int w, int h, String fn, String ftype, int[] LUT)
	{
		int ibuf[] = new int[w*h];
		
		for (int i=0;i<w*h;i++)
			ibuf[i] = LUT[buf[i]];
		
		DataBufferInt dbi = new DataBufferInt(ibuf,w*h);
		SinglePixelPackedSampleModel sppsm = new SinglePixelPackedSampleModel 
			(DataBuffer.TYPE_INT, w,h,w,
		  new int[] {0x00FF0000,0x0000FF00,0x000000FF,0xFF000000});
		Raster r = Raster.createRaster(sppsm,dbi,new Point(0,0));
	
		BufferedImage I = new BufferedImage(w,h, BufferedImage.TYPE_INT_ARGB);
		I.setData(r);
		try
		{
			ImageIO.write(I, ftype, new File(fn));
		}
		catch (IOException e)
		{
			System.err.println(e);
		}
	}
	
	static public void writeViaJIU(IntegerImage I,String fn)
	{
		PNGCodec codec = new PNGCodec();
		try
		{
			codec.setFile(fn, CodecMode.SAVE);
			codec.setImage(I); 
			codec.process();
			codec.close();
		} catch(Exception e)
		{
			e.printStackTrace();
		} 
	}
	
	static public void writeViaJIU(JIUIntegerImage_Image3PixelAccess_Adaptor I, String fn)
	{
		IntegerImage II = I.getUnderlyingImage();
		
		PNGCodec codec = new PNGCodec();
		try
		{
			codec.setFile(fn, CodecMode.SAVE);
			codec.setImage(II); 
			codec.process();
			codec.close();
		} catch(Exception e)
		{
			e.printStackTrace();
		} 
	}
	
	// assumes a scalar, 8-bit 3d volume we're pulling from
	static public void writeViaJIU(Image3PixelAccess I, String fn, int axis, int slice)
	{
		int w,h,d,ww,hh; 
		w = I.getWidth();
		h = I.getHeight();
		d = I.getDepth();
		
		MemoryGray8Image G; 
		
		if (axis == Image3PixelAccess.SLICE_XY)
		{
			ww = w;
			hh = h;
			G = new MemoryGray8Image(ww,hh);
			
			for (int y=0;y<hh;y++)
				for (int x=0;x<ww;x++)
				{
					int v = I.getPixelInt(x, y, slice);
					G.putSample(x,y,v);
				}
		}
		else if (axis == Image3PixelAccess.SLICE_ZY)
		{
			ww = d;
			hh = h;
			G = new MemoryGray8Image(ww,hh);
			
			for (int y=0;y<hh;y++)
				for (int x=0;x<ww;x++)
				{
					int v = I.getPixelInt(slice, y, x);
					G.putSample(x,y,v);
				}
		}
		else
		{
			ww = w;
			hh = d;
			G = new MemoryGray8Image(ww,hh);
			
			for (int y=0;y<hh;y++)
				for (int x=0;x<ww;x++)
				{
					int v = I.getPixelInt(x, slice, y);
					G.putSample(x,y,v);
				}
		}
			
		PNGCodec codec = new PNGCodec();
		try
		{
			codec.setFile(fn, CodecMode.SAVE);
			codec.setImage(G); 
			codec.process();
			codec.close();
		} catch(Exception e)
		{
			e.printStackTrace();
		} 
	}
	
	static public void writeRaw(ScalarImageF3 I, String FN)
	{
		try
		{
			DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(FN)));
			float []buf = I.getBuffer();
			// i don't know how to write the float buffer out as one big chunk in Java
			for (int i=0;i<buf.length;i++)
				dos.writeFloat(buf[i]);
			dos.close();
		}
		catch (IOException e)
		{
			throw new AssertionError(e);
		}
	}
	







	///// LONI IMAGE IO FUNCTIONS TAKEN FROM THEIR EXAMPLES (modified)

  /**
   * Gets an Image Reader that can decode the specified file.
   *
   * @param file File to find an Image Reader for.
   *
   * @return Image Reader that can decode the specified file, or null if none
   *         was found.
   *
   * @throws IOException If an I/O error occurs.
   */
  static private ImageReader _getImageReader(File file) throws IOException
    {
      Object input = new FileImageInputStream(file);
      Object origInput = input;

      // First look for a magic number in the header
      Iterator iter = ImageIO.getImageReaders(input);

      // No Image Reader is found
      if ( !iter.hasNext() ) {

	// Then look in the directory of the file (e.g., ANALYZE, AFNI)
	input = file;
	iter = ImageIO.getImageReaders(input);

	// No Image Reader is found
	if( !iter.hasNext() ) {
	  input = origInput;

	  // Then look at the file name suffix
	  String temp = file.getName();
	  String[] strings = temp.split("\\.");
	  if (strings.length > 1){
	    iter = ImageIO.getImageReadersBySuffix(strings[strings.length-1]);
	  }

	  // No Image Reader found
	  if ( !iter.hasNext() ) { return null; }
	}
      }

      // Set the Input Stream of the first Image Reader returned
      ImageReader imageReader = (ImageReader)iter.next();
      imageReader.setInput(input);

      // Return the Image Reader
      return imageReader;
    }

  /**
   * Gets an Image Writer that can write images read by the specified Image
   * Reader.
   *
   * @param imageReader Image Reader to find an Image Writer for.
   * @param baseFileName Base name used for the output files.
   *
   * @return Image Writer that can write images or null if none was found.
   *
   * @throws IOException If an I/O error occurs.
   */
  static private ImageWriter _getImageWriter(ImageReader imageReader,
				      String baseFileName) throws IOException
    {
      // Find an Image Writer for the Image Reader
      ImageWriter imageWriter = ImageIO.getImageWriter(imageReader);
      if (imageWriter == null) { return null; }

      // Look through the allowed output types
      ImageWriterSpi writerSpi = imageWriter.getOriginatingProvider();
      Class[] outputTypes = writerSpi.getOutputTypes();
      for (int i = 0; i < outputTypes.length; i++) {

	// One output file
	if ( outputTypes[i] == ImageOutputStream.class ) {
	  File outputFile = new File(baseFileName);
	  imageWriter.setOutput( new FileImageOutputStream(outputFile) );
	  return imageWriter;
	}

	// Two output files (AFNI, ANALYZE, ...)
	if ( outputTypes[i] == Vector.class ) {
	  Vector outputStreams = new Vector();

	  // Create HDR and IMG output files
	  File hdrFile = new File(baseFileName + ".hdr");
	  File imgFile = new File(baseFileName + ".img");
	  outputStreams.add( new FileImageOutputStream(hdrFile) );
	  outputStreams.add( new FileImageOutputStream(imgFile) );

	  imageWriter.setOutput(outputStreams);
	  return imageWriter;
	}
      }

      // Unsupported output types
      return null;
    }
  
  static private ImageWriter _getAnalyzeImageWriter(String baseFileName) throws IOException
  {
	  Iterator<ImageWriter> it = ImageIO.getImageWritersByFormatName("analyze");
	  if (!it.hasNext())
	  {
		  System.err.println("Could not find the Analyze PLUGIN for writing...");
		  return null;
	  }
		 
	  
//	  Find an Image Writer for the Image Reader
	  ImageWriter imageWriter = it.next();
	  if (imageWriter == null) { return null; }
	  
//	  Look through the allowed output types
	  ImageWriterSpi writerSpi = imageWriter.getOriginatingProvider();
	  Class[] outputTypes = writerSpi.getOutputTypes();
	  for (int i = 0; i < outputTypes.length; i++) {
		  
//		  One output file  ---> should not occur in this function since it is analyze
		  if ( outputTypes[i] == ImageOutputStream.class ) {
			  File outputFile = new File(baseFileName);
			  imageWriter.setOutput( new FileImageOutputStream(outputFile) );
			  return imageWriter;
		  }
		  
//		  Two output files (AFNI, ANALYZE, ...)
		  if ( outputTypes[i] == Vector.class ) {
			  Vector outputStreams = new Vector();
			  
//			  Create HDR and IMG output files
			  File hdrFile = new File(baseFileName + ".hdr");
			  File imgFile = new File(baseFileName + ".img");
			  outputStreams.add( new FileImageOutputStream(hdrFile) );
			  outputStreams.add( new FileImageOutputStream(imgFile) );
			  
			  imageWriter.setOutput(outputStreams);
			  return imageWriter;
		  }
	  }
	  
//	  Unsupported output types
	  return null;
  }

  static public void displayMetadata(Node root) {
		displayMetadata(root, 0);
	}

	static void indent(int level) {
		for (int i = 0; i < level; i++) {
			System.out.print("  ");
		}
	} 

	static void displayMetadata(Node node, int level) {
		indent(level); // emit open tag
		System.out.print("<" + node.getNodeName());
		NamedNodeMap map = node.getAttributes();
		if (map != null) { // print attribute values
			int length = map.getLength();
			for (int i = 0; i < length; i++) {
				Node attr = map.item(i);
				System.out.print(" " + attr.getNodeName() +
				                 "=\"" + attr.getNodeValue() + "\"");
			}
		}

		Node child = node.getFirstChild();
		if (child != null) {
			System.out.println(">"); // close current tag
			while (child != null) { // emit child tags recursively
				displayMetadata(child, level + 1);
				child = child.getNextSibling();
			}
			indent(level); // emit close tag
			System.out.println("</" + node.getNodeName() + ">");
		} else {
			System.out.println("/>");
		}
	}
  
  
  
}
