package edu.ucla.ccb.graphshifts.image;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentSampleModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.Raster;

import edu.ucla.ccb.graphshifts.data.Point3D;


public class ScalarImageF3  implements Image3PixelAccess
{
	private float[] data_;  // stored band-major, row-sub-major //i.e. z-y-x
	private int[] dim_;	// width, height, depth
	private int depthPlaneSize_;
	private int rowSize_;
	/** the minimum value found in the data_ buffer */
	private float minVal_=Float.POSITIVE_INFINITY;  
	/** the maximum value found in the data_ buffer */
	private float maxVal_=Float.NEGATIVE_INFINITY;
	private boolean imageIsStaticBool = false;
	private int sliceOrientation_ = Image3Protocol.SLICE_XY;
	
	private boolean scaleBySlice = true;


	public ScalarImageF3(int x, int y, int z)
	{
		dim_ = new int[3];
		dim_[0] = x;
		dim_[1] = y;
		dim_[2] = z;	
		depthPlaneSize_ = dim_[0]*dim_[1];
		rowSize_ = dim_[0];

		data_ = new float[z*y*x];
	}


	public void computeMinMax()
	{
		minVal_ = Float.POSITIVE_INFINITY;
		maxVal_ = Float.NEGATIVE_INFINITY;

		for(int z=0;z<dim_[2];z++)
			for(int y=0;y<dim_[1];y++)
				for(int x=0;x<dim_[0];x++)
				{
					float val = getPixelValueAt(x,y,z);
					if (val < minVal_)
						minVal_ = val;
					if (val > maxVal_)
						maxVal_ = val;
				}
	}
	
	final public short getBitsPerPixel()
	{
		return 32;
	}

	public float[] getBuffer()
	{
		return data_;
	}
	
	public Colormap getColormap()
	{
		return null;
	}

	public int getDepth()
	{
		return dim_[2];
	}

	public int getHeight()
	{
		return dim_[1];
	}

	public float getMaxValue()
	{
		if (this.maxVal_ == Float.NEGATIVE_INFINITY)
			computeMinMax();
		return maxVal_;
	}

	public float getMinValue()
	{
		if (this.minVal_ == Float.POSITIVE_INFINITY)
			computeMinMax();
		return minVal_;
	}


	// interface methods for Image3Protocol
	public int getNumberOfSlices()
	{
		if (sliceOrientation_ == Image3Protocol.SLICE_XY)
			return dim_[2];
		else if (sliceOrientation_ == Image3Protocol.SLICE_ZY)
			return dim_[0];
		else 
			return dim_[1];
	}
	
	
	public Object getPixel(int x, int y, int z) {
		return data_[z*depthPlaneSize_+y*rowSize_+x];
	}

	public byte getPixelByte(int x, int y, int z) {
		return (byte)data_[z*depthPlaneSize_+y*rowSize_+x];
	}

	public float getPixelFloat(int x, int y, int z) {
		return (float)data_[z*depthPlaneSize_+y*rowSize_+x];
	}

	public int getPixelInt(int x, int y, int z) {
		return (int)data_[z*depthPlaneSize_+y*rowSize_+x];
	}

	public short getPixelShort(int x, int y, int z) {
		return (short)data_[z*depthPlaneSize_+y*rowSize_+x];
	}


	public float getPixelValueAt(int x, int y, int z)
	{
		return data_[ z*depthPlaneSize_ + y*rowSize_ + x];
	}

	
	public BufferedImage getSlice(int S)
	{
		BufferedImage I;

		if (sliceOrientation_ == Image3Protocol.SLICE_XY)
		{
			byte[] buf = new byte[dim_[0]*dim_[1]];

			if (!scaleBySlice)
			{
				float scale = 255/(maxVal_-minVal_);
				for (int y=0;y<dim_[1];y++)
					for (int x=0;x<dim_[0];x++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(x,y,S)-minVal_)*scale);
						buf[y*dim_[0]+x] = b;
					}
			}
			else
			{
				float sma=Float.NEGATIVE_INFINITY,smi=Float.POSITIVE_INFINITY;
				for (int y=0;y<dim_[1];y++)
					for (int x=0;x<dim_[0];x++)
					{
						float f = getPixelFloat(x, y, S);
						sma = (f > sma) ? f : sma;
						smi = (f < smi) ? f : smi;
					}
				float scale = 255/(sma-smi);
				for (int y=0;y<dim_[1];y++)
					for (int x=0;x<dim_[0];x++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(x,y,S)-smi)*scale);
						buf[y*dim_[0]+x] = b;
					}
			}
	
			DataBufferByte dbb = new DataBufferByte(buf,dim_[0]*dim_[1]);
			ComponentSampleModel csm = new ComponentSampleModel(DataBuffer.TYPE_BYTE,
					dim_[0],dim_[1],1,dim_[0],new int[]{0});
			Point pt = new Point(0,0);
			Raster r = Raster.createRaster(csm,dbb,pt);
	
			I = new BufferedImage(dim_[0],dim_[1],
					                  BufferedImage.TYPE_BYTE_GRAY);
			I.setData(r);
		}
		else if (sliceOrientation_ == Image3Protocol.SLICE_ZY)
		{
			byte[] buf = new byte[dim_[1]*dim_[2]];
			if (!scaleBySlice)
			{
				float scale = 255.0f/(maxVal_-minVal_);
				for (int yy=0;yy<dim_[1];yy++)
					for (int xx=0;xx<dim_[2];xx++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(S,yy,xx)-minVal_)*scale);
						buf[yy*dim_[2]+xx] = b;
					}
			}
			else
			{
				float sma=Float.NEGATIVE_INFINITY,smi=Float.POSITIVE_INFINITY;
				for (int yy=0;yy<dim_[1];yy++)
					for (int xx=0;xx<dim_[2];xx++)
					{
						float f = getPixelFloat(S,yy,xx);
						sma = (f > sma) ? f : sma;
						smi = (f < smi) ? f : smi;
					}
				float scale = 255/(sma-smi);
				for (int yy=0;yy<dim_[1];yy++)
					for (int xx=0;xx<dim_[2];xx++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(S,yy,xx)-smi)*scale);
						buf[yy*dim_[2]+xx] = b;
					}
			}

			DataBufferByte dbb = new DataBufferByte(buf,dim_[2]*dim_[1]);
			ComponentSampleModel csm = new ComponentSampleModel(DataBuffer.TYPE_BYTE,
					dim_[2],dim_[1],1,dim_[2],new int[]{0});
			Point pt = new Point(0,0);
			Raster r = Raster.createRaster(csm,dbb,pt);
	
			I = new BufferedImage(dim_[2],dim_[1], BufferedImage.TYPE_BYTE_GRAY);
			I.setData(r);
		}
		else
		{ // SLICE_XZ
			byte[] buf = new byte[dim_[0]*dim_[2]];
			if (!scaleBySlice)
			{
				float scale = 255.0f/(maxVal_-minVal_);
				for (int yy=0;yy<dim_[2];yy++)
					for (int xx=0;xx<dim_[0];xx++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(xx,S,yy)-minVal_)*scale);
						buf[yy*dim_[0]+xx] = b;
					}
			}
			else
			{
				float sma=Float.NEGATIVE_INFINITY,smi=Float.POSITIVE_INFINITY;
				for (int yy=0;yy<dim_[2];yy++)
					for (int xx=0;xx<dim_[0];xx++)
					{
						float f = getPixelFloat(xx,S,yy);
						sma = (f > sma) ? f : sma;
						smi = (f < smi) ? f : smi;
					}
				float scale = 255/(sma-smi);
				for (int yy=0;yy<dim_[2];yy++)
					for (int xx=0;xx<dim_[0];xx++)
					{
						byte b;
						b = (byte)Math.floor((getPixelValueAt(xx,S,yy)-smi)*scale);
						buf[yy*dim_[0]+xx] = b;
					}
			}

			DataBufferByte dbb = new DataBufferByte(buf,dim_[0]*dim_[2]);
			ComponentSampleModel csm = new ComponentSampleModel(DataBuffer.TYPE_BYTE,
					dim_[0],dim_[2],1,dim_[0],new int[]{0});
			Point pt = new Point(0,0);
			Raster r = Raster.createRaster(csm,dbb,pt);
	
			I = new BufferedImage(dim_[0],dim_[2], BufferedImage.TYPE_BYTE_GRAY);
			I.setData(r);
		}


		return I;
	}
	
	
	final public int getSliceOrientation()
	{
		return sliceOrientation_;
	}


	public int getWidth()
	{
		return dim_[0];
	}

	public boolean hasColormap()
	{
		return false;
	}
	
	public final boolean isValidVoxel(int x, int y, int z)
	{
		if  (    (x < 0) || (x >= getWidth()) ||
				(y < 0) || (y >= getHeight()) ||
				(z < 0) || (z >= getDepth()) )
			return false;
		return true;		
	}
	
	public final boolean isValidVoxel(Point3D P)
	{
		return isValidVoxel((int)(P.getX()),(int)(P.getY()),(int)(P.getZ()));
	}
	
	

	/**
	 * This function should be called when no further modifications to
	 * the underlying image data are permitted.
	 */
	public void makeImageStatic()
	{
		imageIsStaticBool = true;
		computeMinMax();
	}


	public void setPixelAt(int x, int y, int z, float v)
	{
		if (!imageIsStaticBool)
			data_[ z*depthPlaneSize_ + y*rowSize_ + x] = v;
	}

	
	public void setRange(float minv, float maxv)
	{
		this.maxVal_ = maxv;
		this.minVal_ = minv;
	}

	public void setSliceOrientation(int O)
	{
		sliceOrientation_ = O;
	}



}
