package edu.ucla.ccb.graphshifts.observers;

import java.util.ArrayList;

import edu.ucla.ccb.graphshifts.graphs.HardGraphHierarchy;
import edu.ucla.ccb.graphshifts.image.Colormap;
import edu.ucla.ccb.graphshifts.image.Image3PixelAccess;
import edu.ucla.ccb.graphshifts.image.ImageOutliner;
import edu.ucla.ccb.graphshifts.image.ScalarImageB3;
import edu.ucla.ccb.graphshifts.GraphShifts;

/** 
 * This class will only react to certain shift steps.  At these shift steps,
 * it will "wake up" and perform an outlining process based on the current labeling.
 * 
 * The outlines will be written to the specified location with obvious filenames.
 * 
 * @author jcorso
 *
 */
public class ShiftProcessVoyeur implements GraphShiftsObserver
{
	enum TypeOfGrab 
	{
		OUTLINE
		{
			void work(ScalarImageB3 L, WhatToGrab wtg, ImageOutliner io, int[] labelList, String path, int shiftNumber)
			{
				io.setLabels(L);
				io.outline(wtg.slice, wtg.orientation, labelList);
				io.writeLastOutlineImage(String.format("%s-s%06d-%d-%s.png",path,shiftNumber,wtg.slice,wtg.ostr));
			}
		},
		FILL
		{
			void work(ScalarImageB3 L, WhatToGrab wtg, ImageOutliner io, int[] labelList, String path, int shiftNumber)
			{
				io.setLabels(L);
				io.fill(wtg.slice, wtg.orientation, labelList);
				io.writeLastOutlineImage(String.format("%s-s%06d-%d-%s.png",path,shiftNumber,wtg.slice,wtg.ostr));
			}
		},
		GZIPVOLUME
		{
			void work(ScalarImageB3 L, WhatToGrab wtg, ImageOutliner io, int[] labelList, String path, int shiftNumber)
			{
				String name = String.format("%s-s%06d.gzm",path,shiftNumber);
//				System.out.println("writing "+name);
				L.writeToGzipMatrix(name);
			}
		};
		
		abstract void work(ScalarImageB3 L, WhatToGrab wtg, ImageOutliner io, int[] labelList, String path, int shiftNumber);
	}
	
	class WhatToGrab 
	{
		public int slice;
		public String ostr;
		public TypeOfGrab tog;
		public ImageOutliner.Orientation orientation;
		public WhatToGrab(String ostr)
		{
			this.ostr = ostr;
			if (ostr.equals("xy"))
			{
				tog = TypeOfGrab.OUTLINE;
				orientation = ImageOutliner.Orientation.XY;
			}
			else if (ostr.equals("xz"))
			{
				tog = TypeOfGrab.OUTLINE;
				orientation = ImageOutliner.Orientation.XZ;
			}
			else if (ostr.equals("zy"))
			{
				tog = TypeOfGrab.OUTLINE;
				orientation = ImageOutliner.Orientation.ZY;
			}
			else if (ostr.equals("zv"))
			{
				tog = TypeOfGrab.GZIPVOLUME;
			}
			else if (ostr.equals("fxy"))
			{
				tog = TypeOfGrab.FILL;
				orientation = ImageOutliner.Orientation.XY;
			}
			else if (ostr.equals("fxz"))
			{
				tog = TypeOfGrab.FILL;
				orientation = ImageOutliner.Orientation.XZ;
			}
			else if (ostr.equals("fzy"))
			{
				tog = TypeOfGrab.FILL;
				orientation = ImageOutliner.Orientation.ZY;
			}
		}
	}

	Image3PixelAccess raw;
	String path;
	
	Colormap cmap;
	int [] shifts;
	
	int shiftOn=0;
	
	ImageOutliner io;
	
	int[] labelList;
	
	ArrayList<WhatToGrab> wtg;
	
	/**
	 *
	 * @param image
	 * @param cmapPath
	 * @param outputPath
	 * @param shiftsToGrab comma separated list of shifts to grab e.g. "5,10,25"
	 *                     It is assumed that this list is sorted.
	 *                     or, you can use a single -1
	 * @param whatToGrab  comma separate list of what to grab each time 
	 *                    structured in orientation,slice pairs:  xy,65,zy,110
	 */
	public ShiftProcessVoyeur(Image3PixelAccess image, String cmapPath, String outputPath, 
			                  String shiftsToGrab, String whatToGrab, String labelsToGrab)
	{
		this.raw = image;
		this.cmap = Colormap.createFromFile(cmapPath);
		this.path = outputPath;
		
		io = new ImageOutliner(raw,cmap);
		
		buildShiftsToGrab(shiftsToGrab);
		buildWhatToGrab(whatToGrab);
		buildLabelList(labelsToGrab);
	}
	
	private void buildLabelList(String S)
	{
		String []A = S.split(",");
		labelList = new int[A.length];
		for (int i=0;i<A.length;i++)
			labelList[i] = Integer.parseInt(A[i]);
	}
	
	private void buildShiftsToGrab(String S)
	{
		String []A = S.split(",");
		
		shifts = new int[A.length];
		
		for (int i=0;i<A.length;i++)
			shifts[i] = Integer.parseInt(A[i]);
		
	}
	
	private void buildWhatToGrab(String S)
	{
		String []A = S.split(",");
		
		wtg = new ArrayList<WhatToGrab>();
		
		int i=0;
		while (i<A.length)
		{
			WhatToGrab w = new WhatToGrab(A[i++]);
			if ( (w.tog == TypeOfGrab.OUTLINE) || (w.tog == TypeOfGrab.FILL) )
				w.slice = Integer.parseInt(A[i++]);
			
			wtg.add(w);
		}
	}
	
	
	public void shiftBegin()
	{
	}
	
	public void shiftEnd()
	{
	}
	
	public void shiftTaken(int shiftNumber, int shiftLevel, int shiftFrom, int shiftTo, 
			               float shiftWeight, HardGraphHierarchy hierarchy, Object shifter)
	{
		if ((shifter instanceof GraphShifts) && 
		    ((shiftOn < shifts.length) && (shifts[shiftOn] == shiftNumber) ||
		     (shifts[0] == -1))
		   )
		{
			GraphShifts sc = (GraphShifts)shifter;
			
			ScalarImageB3 L = sc.getFinalLabels();
			
			for (WhatToGrab W : wtg)
				W.tog.work(L, W, io, labelList, path, shiftNumber);
		
			shiftOn++;
		}
	}
}