package edu.ucla.ccb.graphshifts.observers;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

import edu.ucla.ccb.graphshifts.graphs.HardGraphHierarchy;
import edu.ucla.ccb.graphshifts.image.Image3PixelAccess;
import edu.ucla.ccb.graphshifts.image.ImageUtilities;
import net.sourceforge.jiu.codecs.CodecMode;
import net.sourceforge.jiu.codecs.InvalidFileStructureException;
import net.sourceforge.jiu.codecs.PNGCodec;
import net.sourceforge.jiu.codecs.UnsupportedCodecModeException;
import net.sourceforge.jiu.codecs.UnsupportedTypeException;
import net.sourceforge.jiu.codecs.WrongFileFormatException;
import net.sourceforge.jiu.data.MemoryRGB24Image;
import net.sourceforge.jiu.ops.MissingParameterException;
import net.sourceforge.jiu.ops.OperationFailedException;

/** This class is a GraphShiftsObserver that exports png image for the top-level segmentation of
 * the graph hierarchy after each shift.
 * @author jcorso
 *
 */
public class ShiftSequenceMaker implements GraphShiftsObserver
{
	Image3PixelAccess image;
	float transparency;

	int slice;
	String path;
	MemoryRGB24Image RGB;
	int cmap[];
	int interval=50;
	public ShiftSequenceMaker(Image3PixelAccess image, float transparency, String path, int number)
	{
		this(image,0,transparency,path,number,50);
	}
	public ShiftSequenceMaker(Image3PixelAccess image, int slice, float transparency, String path,int number)
	{
		this(image,0,transparency,path,number,50);
	}
	
	public ShiftSequenceMaker(Image3PixelAccess image, int slice, float transparency, String path,int number,int interval)
	{
		this.image = image;
		this.slice = slice;
		this.transparency = transparency;
		this.path = path;
		this.interval = interval;
		
		RGB = new MemoryRGB24Image(image.getWidth(),image.getHeight());
		
		// generate a random colormap with "number" entries
		try
		{
			cmap = new int[number];
			BufferedReader re = new BufferedReader(new FileReader("/data/Tu-lonidata/labels/subcortical.lut"));
			int alpha = (int)(255*transparency);
			for(int i=0;i<number;i++)
			{
				String line = re.readLine();
				String []values = line.split("\\s");
				int r,g,b;
				r = Integer.parseInt(values[1]);
				g = Integer.parseInt(values[2]);
				b = Integer.parseInt(values[3]);
				cmap[i] = ((alpha<<24) | (r<<16) | (g<<8) | b);
			}
			re.close();
		} catch (Exception e)
		{
			System.err.println("non-fatal error in trying to load the subcortical lookup table.\n");
			System.err.println("generating a random one instead. Error info follows\n");
			e.printStackTrace();
			cmap = ImageUtilities.makeRandomizedColorMap(number,255);
		} 
	}
	public void setRGBSample(int packed, int voxel, float alpha, int x, int y)
	{
		int sample;
		sample = (int)((1.0f-alpha)*(float)voxel + alpha*(float)((0x00FF0000 & packed) >> 16));
		RGB.putSample(MemoryRGB24Image.INDEX_RED,x,y,   sample);
		sample = (int)((1.0f-alpha)*(float)voxel + alpha*(float)((0x0000FF00 & packed) >> 8));
		RGB.putSample(MemoryRGB24Image.INDEX_GREEN,x,y, (0x0000FF00 & packed) >> 8);
		sample = (int)((1.0f-alpha)*(float)voxel + alpha*(float)((0x000000FF & packed) ));
		RGB.putSample(MemoryRGB24Image.INDEX_BLUE,x,y,  (0x000000FF & packed) );
	}
	
	
	public void shiftBegin()
	{
	}
	
	public void shiftEnd()
	{
	}
	
	public void shiftTaken(int shiftNumber, int shiftLevel, int shiftFrom, int shiftTo, 
			               float shiftWeight, HardGraphHierarchy graphHierarchy, Object shifter)
	{
		if (shiftNumber % interval == 0)
		{
			int labels[] = graphHierarchy.getLabelingFromLevel(graphHierarchy.getNumberOfLayers()-1);

			// all we care about are the class labels at the finest level in the graphHierarchy
			// assume the proper ordering of the nodes onthe finest level in the graphHierarchy
			int w = image.getWidth();
			int h = image.getHeight();
			int wh = w*h;
			for (int y=0;y<h;y++)
				for (int x=0;x<w;x++)
				{
					int label = labels[slice*wh+y*w+x];
					int voxel = image.getPixelInt(x, y, slice);
					setRGBSample(cmap[label],voxel,transparency,x,y);
				}

			writeImage(String.format(path,shiftNumber));
		}
	}
	
	public void writeImage(String path)
	{
		try {
			PNGCodec codec = new PNGCodec();
			codec.setFile(path, CodecMode.SAVE);
			codec.setImage(RGB); 
			codec.process();
			codec.close();
		} catch (UnsupportedCodecModeException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InvalidFileStructureException e) {
			e.printStackTrace();
		} catch (MissingParameterException e) {
			e.printStackTrace();
		} catch (UnsupportedTypeException e) {
			e.printStackTrace();
		} catch (WrongFileFormatException e) {
			e.printStackTrace();
		} catch (OperationFailedException e) {
			e.printStackTrace();
		}
	}
	
}