package edu.buffalo.cse510.hibernate;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.LockMode;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Example;

public abstract class GenericDAO<T, ID extends Serializable> {

  private Class<T> persistentClass;
  private Session session;

  /*
   *  IoC: provide the session you wish to work with in this DAO.
   */
  public GenericDAO(Session session) {

    this();
    this.session = session;
  }

  @SuppressWarnings("unchecked")
  public GenericDAO() {

    this.persistentClass = (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass())
        .getActualTypeArguments()[0];
  }

  protected Class<T> getPersistentClass() {

    return persistentClass;
  }

  protected Session getSession() {

    if (session == null)
      session = HibernateUtil.getSessionFactory().getCurrentSession();
    return session;
  }

  /**
   * Use this inside subclasses as a convenience method.
   */
  @SuppressWarnings("unchecked")
  protected List<T> findByCriteria(Criterion... criterion) {

    Criteria crit = getSession().createCriteria(getPersistentClass());
    for (Criterion c : criterion) {
      crit.add(c);
    }
    return crit.list();
  }

  public List<T> findAll() {

    return findByCriteria();
  }

  @SuppressWarnings("unchecked")
  public List<T> findByExample(T exampleInstance, String... excludeProperty) {

    Criteria crit = getSession().createCriteria(getPersistentClass());
    Example example = Example.create(exampleInstance);
    for (String exclude : excludeProperty) {
      example.excludeProperty(exclude);
    }
    crit.add(example);
    return crit.list();
  }

  @SuppressWarnings("unchecked")
  public T findById(ID id, boolean lock) {

    if (lock)
      return (T) getSession().load(getPersistentClass(), id, LockMode.UPGRADE);
    else
      return findById(id);
  }

  @SuppressWarnings("unchecked")
  public T findById(ID id) {

    return (T) getSession().load(getPersistentClass(), id);
  }

  /**
   * Object States
   * -------------
   * transient: never persistent, not associated with any Session
   * persistent: associated with a unique Session
   * detached: previously persistent, not associated with any Session
   */

  /**
   * Attaches a persistent object not associated with any session to the current session. 
   */
  public void reattach(T entity) {

    getSession().lock(entity, LockMode.NONE);
  }

  /**
   * Makes a transient object persistent or updates a modified persistent object. The flush
   * mode determines at which points the persistent object is flushed to the database. The
   * commit on the transaction makes the inserts/updates permanent.
   */
  public T makePersistent(T entity) {

    getSession().saveOrUpdate(entity);
    return entity;
  }

  /**
   * Makes the object transient, by removing it from the persistence context. The flush mode
   * determines at which points the delete is executed on the database and the commit on the
   * transaction makes the delete permanent.
   */
  public void makeTransient(T entity) {

    getSession().delete(entity);
  }
}
