
/** ----------------------------------------------------------------------
 File "Stock.h", by Qing Liu, referencing Dr.Kenneth Regan's source code in 2009
 *///---------------------------------------------------------------------


#ifndef STOCK_H_
#define STOCK_H_

#include <vector>
#include <string>
#include <iostream>
#include <sstream>
#include <cmath>   

using std::vector;
using std::string;
using std::ostringstream;

struct JavaHashFun {
   size_t operator() (const string& key) {
      size_t hv = 0;
      string::const_iterator keyend = key.end();
      for (string::const_iterator itr = key.begin(); itr != keyend; ++itr) {
         hv = 31*hv + *itr;
      }
      return hv;
   }
};


class Stock {
   friend struct StockHashCode;
   string tickerSymbol;
   size_t sharesOutstanding;
   float lastClose;
   float currentPrice;
   size_t volume;
   bool asc;
   size_t trendTrades;
   size_t trendShares;

   size_t hashCode;
   
public: 
  Stock();
  explicit Stock(const string symbol); 
  Stock(const string symbol, const float openingPrice,
        const size_t givenSharesOutstanding);

 
  Stock(const string stockName, const string stockData);


  virtual ~Stock(); 

  bool match(const Stock* const rhs) const;


  string getSymbol() const; 
  float getPrice() const;
  float getPriceChange() const;
  float getPercentChange() const; 
  float getPercentChangeAbsolute() const;
  size_t getVolume() const; 
  size_t getSharesOutstanding() const;
  float getMomentum() const;
  size_t getTrendTrades() const;
  size_t getTrendShares() const;

  void setName(const string stockName);
  void setPrice(const float stockPrice);
  void setOpeningPrice(const float stockPrice);
  void setVolume(const int stockVolume);
  void update(const string tradeData);

  void update(const size_t sharesTraded, const float atPrice);


   static const int TICK_LENGTH = 5;       
   static const int MOMENTUM_SCALE = 100;  
   static const int VOLUME_SCALE = 1;      
   static const int VOLUME_DIGITS = 5;  
   static string padding(const string datum, const int targetLength);
   static string fmtName(const string tickerSymbol); //assumed <= 5 chars.
   static string fmtUnsignedDecimal(const double value, const int digitsBeforeDot);
   static string fmtSignedDecimal(const double value, const int digitsBeforeDot);

   string str() const;
}; 

inline string Stock::getSymbol() const { return tickerSymbol; }
inline float Stock:: getPrice() const { return currentPrice; }
inline float Stock::getPriceChange() const { return currentPrice - lastClose; }
inline float Stock::getPercentChange() const {
   return (float) 100.0*getPriceChange() / lastClose;
}
inline float Stock::getPercentChangeAbsolute() const {
   return std::fabs(getPercentChange());
}
inline size_t Stock::getVolume() const { return volume; }
inline size_t Stock::getSharesOutstanding() const { return sharesOutstanding; }
inline float Stock::getMomentum() const {
   return getPercentChangeAbsolute()*volume/(float)sharesOutstanding;
}
inline size_t Stock::getTrendTrades() const { return trendTrades; }
inline size_t Stock::getTrendShares() const { return trendShares; }


//-------inline the bellwether update method -------- 

inline  
void Stock::update(const string tradeData) {
   int i = tradeData.find('s');
   istringstream iss(trim(tradeData.substr(0,i)));
   size_t sharesTraded;
   float atPrice;
   iss >> sharesTraded;
   istringstream iss2(trim(tradeData.substr(i+1)));
   iss2 >> atPrice;
   update(sharesTraded, atPrice);
}

inline
void Stock::update(const size_t sharesTraded, const float atPrice) {
   volume += sharesTraded;
   if (atPrice < currentPrice) {
      if (asc) {
         asc = false;
         trendTrades = 1;
         trendShares = sharesTraded;
      } else {
         trendTrades++;
         trendShares += sharesTraded;
      }
   } else if (atPrice > currentPrice) {
      if (asc) {
         trendTrades++;
         trendShares += sharesTraded;
      } else {
         asc = true;
         trendTrades = 1;
         trendShares = sharesTraded;
      }
   } else {  //figure price was unchanged
         trendTrades++;
         trendShares += sharesTraded;
   }
   currentPrice = atPrice;
}

inline
Stock::Stock(const string stockName)
 : tickerSymbol(stockName)
 , sharesOutstanding(0)
 , lastClose(0.0)
 , currentPrice(0.0)
 , volume(0)
 , asc(true)   
 , trendTrades(0)
 , trendShares(0)
 , hashCode(JavaHashFun()(stockName))
{ }


struct StockHash {
   size_t operator() (const Stock& arg) const {
      string key = arg.getSymbol();
      return JavaHashFun()(key);
   }
};

struct StockHashCode {
   size_t operator() (const Stock& arg) const {
      return arg.hashCode;
   }
};

struct StockLess {
   bool operator() (const Stock& lhs, const Stock& rhs) const {
      return lhs.getSymbol() < rhs.getSymbol();
   }
};

struct StockKeyMatch {
   bool operator()(const Stock& lhs, const Stock& rhs) const {
      return (lhs.getSymbol() == rhs.getSymbol());
   }
};

struct StockMatch {
   bool operator()(const Stock& lhs, const Stock& rhs) const {
      return (lhs.match(&rhs));
   }
};



struct CompareVolume {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return (lhs->getVolume() < rhs->getVolume());
   }
};

struct ComparePercentUp {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return (lhs->getPercentChange() < rhs->getPercentChange());
   }
};

struct ComparePercentDown {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return (lhs->getPercentChange() > rhs->getPercentChange());
   }
};

struct ComparePercentChange {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return lhs->getPercentChangeAbsolute() < rhs->getPercentChangeAbsolute();
   }
};

struct CompareMomentum {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return lhs->getMomentum() < rhs->getMomentum();
   }
};

struct CompareTrendTrades {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return lhs->getTrendTrades() < rhs->getTrendTrades();
   }
};

struct CompareTrendShares {
   bool operator()(const Stock* const lhs, const Stock* const rhs) const {
      return lhs->getTrendShares() < rhs->getTrendShares();
   }
};



#endif

