/*
 * Copyright 1999 by Brown University Computer Science
 * 115 Waterman Street, 4th Floor, Providence, Rhode Island, 02906, U.S.A
 * All rights reserved.
 *
 * Permission is hereby granted for free use and modification of this
 * software.  However, this copyright must remain at all times.
 * Re-distribution is restricted to those with the express written consent
 * of the copyright holder.  Permission can be obtained by mailing
 * cs015headtas@cs.brown.edu.  
 */

package NGP.Containers;

//an import statement means I can refer to a class as if I specified the
//whole package
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Component;
import java.awt.Panel;
import java.awt.Dimension;

/**
 * A graphical container that will organize objects in a grid.
 * This class is a subclass of NGP.Container and uses the GridLayout
 * as its LayoutManager.  This means that all objects that use this as a
 * a container end up being sized the same to align correctly.  It is not
 * always the best way of doing things, but should be used when trying to
 * size various components evenly.  You must know how many items will be
 * stored in the grid before you instantiate it.
 * <p>
 * This is especially useful when laying out sliders and labels, essentially
 * Components that you want layed out evenly near each other.
 *
 * @see NGP.Containers.Row
 * @see NGP.Containers.Column
 *
 * @author Matt Chotin (<a href="mailto:mhc@cs.brown.edu">mhc</a>)
 */

public class GridContainer extends NGP.Containers.Panel
  implements NGP.HAligner {

  /** The layout for this container */
  protected GridLayout _layout;
  /** The layout for the panels we will contain */
  private FlowLayout _panelLayout;
  /** How we will store the panels */
  private java.util.Vector _panels;

  /**
   * Construct a grid.  The default alignment is centered, so that all
   * components added to this container will be centered within their cell.
   * Use {@link #alignLeft() alignLeft}, {@link #alignRight() alignRight}, or
   * {@link #alignCenter() alignCenter} to gain control over the alignment
   * of the grid.<br>
   * You must know how many objects you want added to the grid before you
   * instantiate it.
   *
   * @param container the graphical container for this grid
   * @param rows an int saying how many rows this grid will have
   * @param cols an int saying how many columns this grid will have
   */
  public GridContainer(NGP.Container container, int rows, int cols) {
    super(container);
    _layout = new GridLayout(rows, cols);
    this.setLayout(_layout);
    _panelLayout = new FlowLayout(FlowLayout.CENTER);
    _panels = new java.util.Vector();
  }

  /**
   * Normal users need not use this!
   * <p>
   * Add a component to this container.  This will create a panel for each
   * component that is added to the container (to make sure that the sizing
   * of each contained component is optimal).
   *
   * @param comp the component to be added
   * @return returns a Component; you generally do not need to do anything with
   * this
   */
  public Component add(Component comp) {
    Panel panel = new Panel(_panelLayout);
    _panels.addElement(panel);
    panel.add(comp);
    return super.add(panel);
  }

  /**
   * Remove the component from the container and resize the container.
   * @param comp the component to remove
   */
  public void remove(Component comp) {
    Panel panel = null;

    //go through the vector of panels and get the one that contains the
    //component we want to remove

    for (int i = 0; i < _panels.size(); i++) {
      panel = (Panel)_panels.get(i);
      if (comp == panel.getComponent(0)) break;
      panel = null;
    }

    //make sure we found the panel, then remove the component and resize
    //the grid

    if (panel != null) {
      panel.remove(comp);
      _panels.removeElement(panel);
      super.remove(panel);
      this.setSize(this.getPreferredSize());
    }
  }

  /**
   * Align all components added after this is called to the left.
   * Will not affect any components added previously to this call.
   */
  public void alignLeft() {
    _panelLayout = new FlowLayout(FlowLayout.LEFT);
  }

  /**
   * Align all components added after this is called to the right.
   * Will not affect any components added previously to this call.
   */
  public void alignRight() {
    _panelLayout = new FlowLayout(FlowLayout.RIGHT);
  }

  /**
   * Align all components added after this is called to the center.
   * Will not affect any components added previously to this call.
   */
  public void alignCenter() {
    _panelLayout = new FlowLayout(FlowLayout.CENTER);
  }

}
