/*
 * Copyright 1999 by Brown University Computer Science
 * 115 Waterman Street, 4th Floor, Providence, Rhode Island, 02906, U.S.A
 * All rights reserved.
 *
 * Permission is hereby granted for free use and modification of this
 * software.  However, this copyright must remain at all times.
 * Re-distribution is restricted to those with the express written consent
 * of the copyright holder.  Permission can be obtained by mailing
 * cs015headtas@cs.brown.edu.  
 */

package NGP;

/**
 * An object that "activates" at a specified interval.  This class can be
 * used to help model both an "Animator" that might affect an entire program,
 * or a "Behavior" that might affect only an individual object.  The
 * {@link #activate() activate} method will fire every time a certain number
 * of milliseconds has passed.  It is up to the subclass to override
 * <code>activate</code> to determine what should take place.<p>
 *
 * This timer will not stop activating until the program ends or
 * {@link #stop() stop} is called explicitly.
 *
 * @author Matt Chotin (<a href="mailto:mhc@cs.brown.edu">mhc</a>)
 */

public abstract class Timer implements java.awt.event.ActionListener {

  /** the Swing Timer that actually does the timekeeping */
  private javax.swing.Timer _swingTimer;

  /**
   * Creates a <code>Timer</code> with a delay between
   * {@link #activate() activate} calls
   *
   * @param delay milliseconds between activate calls
   */
  public Timer(int delay) {
    _swingTimer = new javax.swing.Timer(delay, this);
  }

  /**
   * Normal users need not deal with this method!
   * <p>
   * Called by the Swing Timer when the time has expired.  It will then call
   * {@link #activate() activate} which subclasses should override.
   *
   * @param e the java.awt.event.ActionEvent that was sent, we do not deal
   * with it
   */
  public void actionPerformed(java.awt.event.ActionEvent e) {
    this.activate();
  }

  /**
   * Called whenever the set time has expired.  Subclasses should override
   * to do something useful.
   */
  public abstract void activate();

  /**
   * Return the number of milliseconds this Timer uses as a delay.
   *
   * @return the number of milliseconds the Timer uses as a delay
   * @see #setDelay(int) setDelay
   */
  public int getDelay() {
    return _swingTimer.getDelay();
  }

  /**
   * Set the number of milliseconds this Timer uses as a delay.
   *
   * @param d the number of milliseconds the Timer should use as a delay
   * @see #getDelay() getDelay
   */
  public void setDelay(int d) {
    _swingTimer.setDelay(d);
  }

  /**
   * Start the timer, a timer is stopped by default.
   *
   * @see #stop() stop
   */
  public void start() {
    _swingTimer.start();
  }

  /**
   * Stop the timer, a timer is stopped by default.
   *
   * @see #start() start
   */
  public void stop() {
    _swingTimer.stop();
  }

}

